<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\PaymentServicesPaypal\Plugin\InstantPurchase;

use Magento\Customer\Model\Session as CustomerSession;
use Magento\PaymentServicesBase\Model\HttpException;
use Magento\PaymentServicesPaypal\Helper\OrderHelper;
use Magento\PaymentServicesPaypal\Model\OrderService;
use Magento\PaymentServicesPaypal\Model\Ui\TokenUiComponentProvider;
use Magento\Quote\Model\Quote;
use Magento\InstantPurchase\Model\QuoteManagement\PaymentConfiguration as InstantPurchasePaymentConfiguration;
use Magento\Framework\Exception\LocalizedException;
use Magento\PaymentServicesBase\Model\Config;
use Magento\PaymentServicesPaypal\Model\HostedFieldsConfigProvider;

class PaymentConfiguration
{
    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var OrderHelper
     */
    private $orderHelper;

    /**
     * @param OrderService $orderService
     * @param CustomerSession $customerSession
     * @param Config $config
     * @param OrderHelper $orderHelper
     */
    public function __construct(
        OrderService $orderService,
        CustomerSession $customerSession,
        Config $config,
        OrderHelper $orderHelper
    ) {
        $this->orderService = $orderService;
        $this->customerSession = $customerSession;
        $this->config = $config;
        $this->orderHelper = $orderHelper;
    }

    /**
     * Create PayPal order on instant purchase.
     *
     * @param InstantPurchasePaymentConfiguration $subject
     * @param Quote $quote
     * @return Quote
     * @throws LocalizedException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterConfigurePayment(
        InstantPurchasePaymentConfiguration $subject,
        Quote $quote
    ): Quote {
        if ($quote->getPayment()->getMethod() !== HostedFieldsConfigProvider::CC_VAULT_CODE) {
            return $quote;
        }

        $this->forceTotalsRecalculation($quote);

        $totalAmount = $quote->getBaseGrandTotal();
        $currencyCode = $quote->getCurrency()->getBaseCurrencyCode();
        $customer = $this->customerSession->getCustomer();
        $response = $this->orderService->create(
            [
                'amount' => $this->orderHelper->formatAmount((float)$totalAmount),
                'l2_data' => $this->orderHelper->getL2Data($quote, TokenUiComponentProvider::CC_VAULT_SOURCE),
                'l3_data' => $this->orderHelper->getL3Data($quote, TokenUiComponentProvider::CC_VAULT_SOURCE),
                'currency_code' => $currencyCode,
                'is_digital' => $quote->getIsVirtual(),
                'website_id' => $quote->getStore()->getWebsiteId(),
                'shipping_address' => $this->orderService->mapAddress($quote->getShippingAddress()),
                'billing_address' => $this->orderService->mapAddress($quote->getBillingAddress()),
                'payer' => $this->orderService->buildPayer($quote, $customer->getId()),
                'payment_source' => TokenUiComponentProvider::CC_VAULT_SOURCE,
                'quote_id' => $quote->getId(),
                'order_increment_id' => $this->orderHelper->reserveAndGetOrderIncrementId($quote)
            ]
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to create an order.');
        }

        $quote->getPayment()
            ->setAdditionalInformation('paypal_order_id', $response['paypal-order']['id'])
            ->setAdditionalInformation('payments_order_id', $response['paypal-order']['mp_order_id'])
            ->setAdditionalInformation('payments_mode', $this->config->getEnvironmentType())
            ->setAdditionalInformation('paypal_order_amount', $totalAmount);

        return $quote;
    }

    /**
     * Force recalculation of quote shipping rates
     *
     * @param Quote $quote
     * @return void
     */
    private function forceTotalsRecalculation(Quote $quote): void
    {
        // Force recalculation of shipping rates to have correct shipping and base grand total in quote
        // This only happens with InstantPurchase checkout
        $quote->setTotalsCollectedFlag(false);

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->getBillingAddress()->setCollectShippingRates(true);

        $quote->collectTotals();
    }
}
