<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

declare(strict_types=1);

namespace Magento\PaymentServicesPaypal\Gateway\Request;

use Magento\Framework\App\Request\Http;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\PaymentServicesPaypal\Model\Config;
use Magento\Payment\Gateway\Data\PaymentDataObjectInterface;
use Magento\Payment\Gateway\Helper\SubjectReader;
use Magento\Payment\Gateway\Request\BuilderInterface;
use Magento\Store\Model\StoreManagerInterface;

class VaultAuthorizationRequest implements BuilderInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param Config $config
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Config $config,
        StoreManagerInterface $storeManager
    ) {
        $this->config = $config;
        $this->storeManager = $storeManager;
    }

    /**
     * Build authorization request for payment with vaulted card
     *
     * @param array $buildSubject
     * @return array
     * @throws NoSuchEntityException
     */
    public function build(array $buildSubject)
    {
        /** @var PaymentDataObjectInterface $payment */
        $paymentDO = SubjectReader::readPayment($buildSubject);
        $payment = $paymentDO->getPayment();
        $extensionAttributes = $payment->getExtensionAttributes();
        $paymentToken = $extensionAttributes->getVaultPaymentToken();

        $uri = '/payments/'
            . $this->config->getMerchantId()
            . '/payment/paypal/order/'
            . $payment->getAdditionalInformation('paypal_order_id')
            . '/authorize';
        $websiteId = $this->storeManager->getStore($paymentDO->getOrder()->getStoreId())->getWebsiteId();

        $request = [
            'uri' => $uri,
            'method' => \Magento\Framework\App\Request\Http::METHOD_POST,
            'body' => [
                'mp-transaction' => [
                    'order-increment-id' => $paymentDO->getOrder()->getOrderIncrementId(),
                    'payment-vault-id' => $paymentToken->getGatewayToken()
                ]
            ],
            'headers' => [
                'Content-Type' => 'application/json',
                'x-scope-id' => $websiteId,
            ]
        ];
        $customerId = $payment->getAdditionalInformation('customer_id');

        if ($customerId) {
            $request['headers']['x-commerce-customer-id'] = $customerId;
        }

        return $request;
    }
}
