<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Gateway\Request;

use Magento\PaymentServicesPaypal\Model\Config;
use Magento\Payment\Gateway\Data\PaymentDataObjectInterface;
use Magento\Payment\Gateway\Helper\SubjectReader;
use Magento\Payment\Gateway\Request\BuilderInterface;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Exception\NoSuchEntityException;

class AuthorizationRequest implements BuilderInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * @param Config $config
     * @param StoreManagerInterface $storeManager
     * @param CustomerSession $customerSession
     */
    public function __construct(
        Config $config,
        StoreManagerInterface $storeManager,
        CustomerSession $customerSession
    ) {
        $this->config = $config;
        $this->storeManager = $storeManager;
        $this->customerSession = $customerSession;
    }

    /**
     * Build authorization request
     *
     * @param array $buildSubject
     * @return array
     * @throws NoSuchEntityException
     */
    public function build(array $buildSubject)
    {
        /** @var PaymentDataObjectInterface $payment */
        $payment = SubjectReader::readPayment($buildSubject);

        $uri = '/payments/'
            . $this->config->getMerchantId()
            . '/payment/paypal/order/'
            . $payment->getPayment()->getAdditionalInformation('paypal_order_id')
            . '/authorize';
        $websiteId = $this->storeManager->getStore($payment->getOrder()->getStoreId())->getWebsiteId();
        $request = [
            'uri' => $uri,
            'method' => \Magento\Framework\App\Request\Http::METHOD_POST,
            'body' => [
                'mp-transaction' => [
                    'order-increment-id' => $payment->getOrder()->getOrderIncrementId()
                ]
            ],
            'headers' => [
                'Content-Type' => 'application/json',
                'x-scope-id' => $websiteId,
            ]
        ];
        $customerId = $this->customerSession->getCustomerId();

        if ($customerId) {
            $request['headers']['x-commerce-customer-id'] = $customerId;
        }

        return $request;
    }
}
