<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Gateway\Http;

use Magento\Payment\Gateway\Http\ClientInterface;
use Magento\Payment\Gateway\Http\TransferInterface;
use Magento\Payment\Gateway\Http\ClientException;
use Magento\PaymentServicesBase\Model\ServiceClientInterface;
use Magento\Payment\Model\Method\Logger;

class ServiceClient implements ClientInterface
{
    private const DENIED_RESPONSE = "PAYMENT_DENIED";

    /**
     * @var ServiceClientInterface
     */
    private $httpClient;

    /**
     * @var Logger
     */
    private $logger;

    /**
     * @param ServiceClientInterface $httpClient
     * @param Logger $logger
     */
    public function __construct(
        ServiceClientInterface $httpClient,
        Logger $logger
    ) {
        $this->httpClient = $httpClient;
        $this->logger = $logger;
    }

    /**
     * Places request to gateway. Returns result as ENV array
     *
     * @param TransferInterface $transferObject
     * @return array
     * @throws \Magento\Payment\Gateway\Http\ClientException
     */
    public function placeRequest(TransferInterface $transferObject)
    {
        $response = $this->httpClient->request(
            $transferObject->getHeaders(),
            $transferObject->getUri(),
            $transferObject->getMethod(),
            $transferObject->getBody() == null ? '' : json_encode($transferObject->getBody())
        );

        $this->logger->debug(
            [
                'request' => [
                    $transferObject->getUri(),
                    $transferObject->getHeaders(),
                    $transferObject->getMethod(),
                    $transferObject->getBody()
                ],
                'response' => $response
            ]
        );

        if (!$response['is_successful']) {
            if ($response['message'] === self::DENIED_RESPONSE) {
                throw new ClientException(
                    __(
                        'Your payment was not successful. '
                        . 'Ensure you have entered your details correctly and try again, '
                        . 'or try a different payment method. If you have continued problems, '
                        . 'contact the issuing bank for your payment method.'
                    )
                );
            } else {
                throw new ClientException(
                    __('Error happened when processing the request. Please try again later.')
                );
            }
        }

        return $response;
    }
}
