<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Model;

use Magento\PaymentServicesBase\Model\ServiceClientInterface;
use Magento\Framework\App\Request\Http;
use Magento\PaymentServicesBase\Model\HttpException;

class OrderService
{
    /**
     * @var ServiceClientInterface
     */
    private $httpClient;

    /**
     * @var Config
     */
    private $config;

    /**
     * @param ServiceClientInterface $httpClient
     * @param Config $config
     */
    public function __construct(
        ServiceClientInterface $httpClient,
        Config $config
    ) {
        $this->httpClient = $httpClient;
        $this->config = $config;
    }

    /**
     * Map DTO fields and send the order creation request to the backend service
     *
     * @param array $data
     * @return array
     * @throws HttpException
     */
    public function create(array $data) : array
    {
        $order = [
            'paypal-order' => [
                'amount' => [
                    'currency_code' => $data['currency_code'],
                    'value' => $data['amount'] ?? 0.00
                ],
                'is_digital' => !!$data['is_digital'] ?? false,
                'website_id' => $data['website_id']
            ]
        ];
        $order['paypal-order']['shipping-address'] = $data['shipping_address'] ?? null;
        $order['paypal-order']['billing-address'] = $data['billing_address'] ?? null;
        $order['paypal-order']['payer'] = $data['payer'] ?? null;
        $softDescriptor = $this->config->getSoftDescriptor($data['store_code'] ?? null);
        if ($softDescriptor) {
            $order['paypal-order']['soft_descriptor'] = $softDescriptor;
        }
        $response = $this->httpClient->request(
            [
                'Content-Type' => 'application/json',
                'x-scope-id' => $data['website_id']
            ],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order',
            Http::METHOD_POST,
            json_encode($order)
        );

        return $response;
    }

    /**
     * Update the PayPal order with selective params
     *
     * @param string $id
     * @param array $data
     * @throws HttpException
     */
    public function update(string $id, array $data) : void
    {
        $order = [
            'paypal-order-update' => [
                'reference_id' => 'default',
                'amount' => [
                    'operation' => 'REPLACE',
                    'value' => [
                        'currency_code' => $data['currency_code'],
                        'value' => $data['amount']
                    ]
                ]
            ]
        ];
        $response = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order/' . $id,
            Http::METHOD_PATCH,
            json_encode($order)
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to update an order.');
        }
    }

    /**
     * Get the Order object from PayPal
     *
     * @param string $id
     * @return array
     * @throws HttpException
     */
    public function get(string $id) : array
    {
        $response = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order/' . $id,
            Http::METHOD_GET,
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to retrieve an order.');
        }
        return $response;
    }
}
