<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\PaymentServicesPaypal\Model\Adminhtml;

use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\PaymentServicesPaypal\Model\SdkService;
use Magento\PaymentServicesPaypal\Model\SdkService\PaymentOptionsBuilderFactory;
use Magento\PaymentServicesPaypal\Model\Config;
use Magento\Framework\Exception\NoSuchEntityException;

use Magento\Framework\DataObject;

class SdkParams extends DataObject
{
    /**
     * @var PaymentOptionsBuilderFactory
     */
    private $paymentOptionsBuilderFactory;

    /**
     * @var SdkService
     */
    private $sdkService;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Config
     */
    private $config;

    /**
     * SdkParams constructor.
     * @param PaymentOptionsBuilderFactory $paymentOptionsBuilderFactory
     * @param SdkService $sdkService
     * @param SerializerInterface $serializer
     * @param Config $config
     * @param array $data
     */
    public function __construct(
        PaymentOptionsBuilderFactory $paymentOptionsBuilderFactory,
        SdkService $sdkService,
        SerializerInterface $serializer,
        Config $config,
        array $data = []
    ) {
        parent::__construct($data);
        $this->paymentOptionsBuilderFactory = $paymentOptionsBuilderFactory;
        $this->sdkService = $sdkService;
        $this->serializer = $serializer;
        $this->config = $config;
    }

    /**
     * Get sdk params.
     *
     * @param string|null $websiteId
     * @return array
     * @throws NoSuchEntityException
     */
    public function getSdkParams($websiteId = null): array
    {
        $paymentOptionsBuilder = $this->paymentOptionsBuilderFactory->create();
        $paymentOptionsBuilder->setAreButtonsEnabled(false);
        $paymentOptionsBuilder->setIsPayPalCreditEnabled($this->config->isFundingSourceEnabledByName('paypal_credit'));
        $paymentOptionsBuilder->setIsVenmoEnabled($this->config->isFundingSourceEnabledByName('venmo'));
        $paymentOptionsBuilder->setIsCreditCardEnabled(true);
        $paymentOptionsBuilder->setIsPaylaterMessageEnabled(false);
        $paymentOptions = $paymentOptionsBuilder->build();
        try {
            $params = $this->sdkService->getSdkParams(
                $paymentOptions,
                false,
                SdkService::PAYMENT_ACTION_AUTHORIZE,
                $websiteId
            );
        } catch (InvalidArgumentException $e) {
            return [];
        }
        $result = [];
        foreach ($params as $param) {
            $result[] = [
                'name' => $param['name'],
                'value' => $param['value']
            ];
        }
        return $result;
    }
}
