<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Controller\Order;

use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\PaymentServicesPaypal\Model\OrderService;
use Magento\PaymentServicesBase\Model\HttpException;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address as Address;

class Create implements HttpPostActionInterface, CsrfAwareActionInterface
{
    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var ResultFactory
     */
    private $resultFactory;

    /**
     * @param CheckoutSession $checkoutSession
     * @param CustomerSession $customerSession
     * @param OrderService $orderService
     * @param ResultFactory $resultFactory
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        CustomerSession $customerSession,
        OrderService $orderService,
        ResultFactory $resultFactory
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->customerSession = $customerSession;
        $this->orderService = $orderService;
        $this->resultFactory = $resultFactory;
    }

    /**
     * Dispatch the order creation request with Commerce params
     *
     * @return ResultInterface
     */
    public function execute() : ResultInterface
    {
        $result = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        try {
            $quote = $this->checkoutSession->getQuote();
            $response = $this->orderService->create(
                [
                    'amount' => number_format($quote->getBaseGrandTotal(), 2, '.', ''),
                    'currency_code' => $quote->getCurrency()->getBaseCurrencyCode(),
                    'shipping_address' => $this->mapAddress($quote->getShippingAddress()),
                    'billing_address' => $this->mapAddress($quote->getBillingAddress()),
                    'payer' => $this->buildPayer($quote),
                    'is_digital' => $quote->isVirtual(),
                    'website_id' => $quote->getStore()->getWebsiteId()
                ]
            );
            $result->setHttpResponseCode($response['status'])
                ->setData(['response' => $response]);
        } catch (HttpException $e) {
            $result->setHttpResponseCode(500);
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function createCsrfValidationException(RequestInterface $request) :? InvalidRequestException
    {
        return null;
    }

    /**
     * @inheritdoc
     */
    public function validateForCsrf(RequestInterface $request) :? bool
    {
        return true;
    }

    /**
     * Map Commerce address fields to DTO
     *
     * @param Address $address
     * @return array|null
     */
    private function mapAddress(Address $address) :? array
    {
        // The PayPal address object requires country_code
        // https://developer.paypal.com/api/orders/v2/#definition-address_name
        if ($address->getCountry() === null) {
            return null;
        }
        return [
            'full_name' => $address->getFirstname() . ' ' . $address->getLastname(),
            'address_line_1' => $address->getStreet()[0],
            'address_line_2' => $address->getStreet()[1] ?? null,
            'admin_area_1' => $address->getRegion(),
            'admin_area_2' => $address->getCity(),
            'postal_code' => $address->getPostcode(),
            'country_code' => $address->getCountry()
        ];
    }

    /**
     * Build the Payer object for PayPal order creation
     *
     * @param Quote $quote
     * @return array
     */
    private function buildPayer(Quote $quote) : array
    {
        $isLoggedIn = $this->customerSession->isLoggedIn();
        $billingAddress = $quote->getBillingAddress();
        $phone = $billingAddress->getTelephone();
        if (!$isLoggedIn) {
            return [
                'name' => [
                    'given_name' => $billingAddress->getFirstname(),
                    'surname' => $billingAddress->getLastname()
                ],
                'email' => $billingAddress->getEmail(),
                'phone_number' => $phone !== null ? preg_replace('/[^0-9]/', '', $phone) : null
            ];
        }

        return [
            'name' => [
                'given_name' => $quote->getCustomerFirstname(),
                'surname' => $quote->getCustomerLastname()
            ],
            'email' => $quote->getCustomerEmail(),
            // PayPal Payer phone_number field has to be entirely numerical
            // https://developer.paypal.com/api/orders/v2/#definition-phone_with_type.phone
            'phone_number' => $phone !== null ? preg_replace('/[^0-9]/', '', $phone) : null
        ];
    }
}
