<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Model;

use Magento\Framework\App\Request\Http;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\ResolverInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\PaymentServicesBase\Model\ServiceClientInterface;
use Magento\Store\Model\StoreManagerInterface;

class SdkService
{
    public const PAYMENT_ACTION_AUTHORIZE = 'authorize';
    private const SDK_ATTRIBUTES = 'sdk-attributes';
    private const SCRIPT_ATTRIBUTES = 'script-attributes';
    private const CURRENCY = 'currency';
    private const LOCALE = 'locale';
    private const PARAM_OBJ_KEY = 'sdk_params';
    private const PAYMENT_OPTIONS = 'payment_options';
    private const IS_REVIEW_PAYMENT_REQUIRED = 'review_payment';
    private const PAYMENT_ACTION = 'payment_action';
    private const PAYMENT_BUILD_SDK_URL_PATH = '/payments/payment/paypal/sdkurl/';

    /**
     * @var Config
     */
    private $config;

    /**
     * @var ServiceClientInterface
     */
    private $httpClient;

    /**
     * @var ResolverInterface
     */
    private $localeResolver;

    /**
     * @var Json
     */
    private $serializer;

    /**
     * @var
     */
    private $storeManager;

    /**
     * SdkService constructor.
     * @param Config $config
     * @param ServiceClientInterface $httpClient
     * @param ResolverInterface $localeResolver
     * @param Json $serializer
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Config $config,
        ServiceClientInterface $httpClient,
        ResolverInterface $localeResolver,
        Json $serializer,
        StoreManagerInterface $storeManager
    ) {
        $this->config = $config;
        $this->httpClient = $httpClient;
        $this->localeResolver = $localeResolver;
        $this->serializer = $serializer;
        $this->storeManager = $storeManager;
    }

    /**
     * @param array $paymentOptions
     * @param bool $isReviewPaymentEnabled
     * @param String $paymentAction
     * @return array|mixed
     * @throws NoSuchEntityException
     */
    public function getSdkParams(
        array $paymentOptions,
        bool $isReviewPaymentEnabled,
        String $paymentAction
    ) {
        $currency = $this->storeManager->getStore()->getBaseCurrencyCode();
        $sdkParams = [
            self::PARAM_OBJ_KEY => [
                self::CURRENCY => $currency,
                self::LOCALE => $this->localeResolver->getLocale(),
                self::PAYMENT_OPTIONS => $paymentOptions,
                self::IS_REVIEW_PAYMENT_REQUIRED => $isReviewPaymentEnabled,
                self::PAYMENT_ACTION => $paymentAction
            ]
        ];
        $result = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            self::PAYMENT_BUILD_SDK_URL_PATH,
            Http::METHOD_POST,
            $this->serializer->serialize($sdkParams)
        );
        if (!$result['is_successful']) {
            return [];
        }
        return $result[self::SDK_ATTRIBUTES][self::SCRIPT_ATTRIBUTES];
    }
}
