<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\PaymentServicesPaypal\CustomerData;

use Magento\Customer\CustomerData\SectionSourceInterface;
use Magento\Framework\DataObject;
use Magento\PaymentServicesPaypal\Model\Config;
use Magento\PaymentServicesPaypal\Model\SdkService;
use Magento\PaymentServicesPaypal\Model\SdkService\PaymentOptionsBuilderFactory;

/**
 * Sdk Params Source
 */
class SdkParams extends DataObject implements SectionSourceInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var PaymentOptionsBuilderFactory
     */
    private $paymentOptionsBuilderFactory;

    /**
     * @var SdkService
     */
    private $sdkService;

    /**
     * @param PaymentOptionsBuilderFactory $paymentOptionsBuilderFactory
     * @param SdkService $sdkService
     * @param Config $config
     * @param array $data
     */
    public function __construct(
        PaymentOptionsBuilderFactory $paymentOptionsBuilderFactory,
        SdkService $sdkService,
        Config $config,
        array $data = []
    ) {
        parent::__construct($data);
        $this->paymentOptionsBuilderFactory = $paymentOptionsBuilderFactory;
        $this->sdkService = $sdkService;
        $this->config = $config;
    }

    /**
     * @inheritdoc
     */
    public function getSectionData()
    {
        return [
            'sdkParams' => $this->getSdkParams()
        ];
    }

    /**
     * Get script params for paypal smart buttons sdk.
     *
     * @return array
     */
    private function getSdkParams() : array
    {
        if (!$this->config->isEnabled()) {
            return [];
        }
        $paymentOptionsBuilder = $this->paymentOptionsBuilderFactory->create();
        $paymentOptionsBuilder->setAreButtonsEnabled(true);
        $paymentOptionsBuilder->setIsPayPalCreditEnabled($this->config->isFundingSourceEnabledByName('paypal_credit'));
        $paymentOptionsBuilder->setIsVenmoEnabled($this->config->isFundingSourceEnabledByName('venmo'));
        $paymentOptionsBuilder->setIsCreditCardEnabled(false);
        $paymentOptionsBuilder->setIsPaylaterMessageEnabled(true);
        $paymentOptions = $paymentOptionsBuilder->build();
        try {
            $params = $this->sdkService->getSdkParams(
                $paymentOptions,
                true,
                SdkService::PAYMENT_ACTION_AUTHORIZE
            );
        } catch (\InvalidArgumentException $e) {
            return [];
        }
        $result = [];
        foreach ($params as $param) {
            $result[] = [
                'name' => $param['name'],
                'value' => $param['value']
            ];
        }
        return $result;
    }
}
