<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Model;

use Magento\PaymentServicesBase\Model\ServiceClientInterface;
use Magento\Framework\App\Request\Http;
use Magento\PaymentServicesBase\Model\HttpException;

class OrderService
{
    /**
     * @var ServiceClientInterface
     */
    private $httpClient;

    /**
     * @var Config
     */
    private $config;

    /**
     * @param ServiceClientInterface $httpClient
     * @param Config $config
     */
    public function __construct(
        ServiceClientInterface $httpClient,
        Config $config
    ) {
        $this->httpClient = $httpClient;
        $this->config = $config;
    }

    /**
     * @param array $data
     * @return array
     * @throws HttpException
     */
    public function create(array $data) : array
    {
        $order = [
            'paypal-order' => [
                'amount' => [
                    'currency_code' => $data['currency_code'],
                    'value' => $data['amount'] ?? 0.00
                ],
                'is_digital' => !!$data['is_digital'] ?? false,
                'website_id' => $data['website_id']
            ]
        ];
        $shippingAddress = $data['shipping_address'] ?? null;
        if ($shippingAddress) {
            $order['paypal-order']['shipping-address'] = [
                'full_name' => $shippingAddress['name'],
                'address_line_1' => $shippingAddress['street'][0],
                'address_line_2' => $shippingAddress['street'][1],
                'admin_area_1' => $shippingAddress['region'],
                'admin_area_2' => $shippingAddress['city'],
                'postal_code' => $shippingAddress['postcode'],
                'country_code' => $shippingAddress['country'],
            ];
        }
        $response = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order',
            Http::METHOD_POST,
            json_encode($order)
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to create an order.');
        }
        return $response;
    }

    /**
     * @param string $id
     * @param array $data
     * @throws HttpException
     */
    public function update(string $id, array $data) : void
    {
        $order = [
            'paypal-order-update' => [
                'reference_id' => 'default',
                'amount' => [
                    'operation' => 'REPLACE',
                    'value' => [
                        'currency_code' => $data['currency_code'],
                        'value' => $data['amount']
                    ]
                ]
            ]
        ];
        $response = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order/' . $id,
            Http::METHOD_PATCH,
            json_encode($order)
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to update an order.');
        }
    }

    /**
     * @param string $id
     * @return array
     * @throws HttpException
     */
    public function get(string $id) : array
    {
       $response = $this->httpClient->request(
            ['Content-Type' => 'application/json'],
            '/payments/' . $this->config->getMerchantId() . '/payment/paypal/order/' . $id,
            Http::METHOD_GET,
        );
        if (!$response['is_successful']) {
            throw new HttpException('Failed to retrieve an order.');
        }
        return $response;
    }
}
