<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Controller\Order;

use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\PaymentServicesPaypal\Model\OrderService;
use Magento\PaymentServicesBase\Model\HttpException;
use Magento\Quote\Model\Quote\Address as Address;

class Create implements HttpPostActionInterface, CsrfAwareActionInterface
{
    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var ResultFactory
     */
    private $resultFactory;

    /**
     * @param CheckoutSession $checkoutSession
     * @param OrderService $orderService
     * @param ResultFactory $resultFactory
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        OrderService $orderService,
        ResultFactory $resultFactory
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->orderService = $orderService;
        $this->resultFactory = $resultFactory;
    }

    /**
      * @return ResultInterface
     */
    public function execute() : ResultInterface
    {
        $result = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        try {
            $quote = $this->checkoutSession->getQuote();
            $response = $this->orderService->create(
                [
                    'amount' => number_format($quote->getBaseGrandTotal(), 2, '.', ''),
                    'currency_code' => $quote->getCurrency()->getBaseCurrencyCode(),
                    'shipping_address' => $this->mapShippingAddress($quote->getShippingAddress()),
                    'is_digital' => $quote->isVirtual(),
                    'website_id' => $quote->getStore()->getWebsiteId()
                ]
            );
            $result->setHttpResponseCode($response['status'])
                ->setData(['response' => $response]);
        } catch (HttpException $e) {
            $result->setHttpResponseCode(500);
        }
        return $result;
    }

    public function createCsrfValidationException(RequestInterface $request) :? InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request) :? bool
    {
        return true;
    }

    private function mapShippingAddress(Address $shippingAddress) :? array
    {
        return [
            'name' => $shippingAddress->getFirstname() . ' ' . $shippingAddress->getLastname(),
            'street' => [
                0 => $shippingAddress->getStreet()[0],
                1 => isset($shippingAddress->getStreet()[1]) ? $shippingAddress->getStreet()[1] : ''
            ],
            'region' => $shippingAddress->getRegion(),
            'city' => $shippingAddress->getCity(),
            'postcode' => $shippingAddress->getPostcode(),
            'country' => $shippingAddress->getCountry()
        ];
    }
}
