<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesPaypal\Controller\Adminhtml\Order;

use Magento\Backend\App\Action\Context;
use Magento\Backend\Model\Session\Quote;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\PaymentServicesBase\Model\HttpException;
use Magento\PaymentServicesPaypal\Model\OrderService;
use Magento\Quote\Model\Quote\Address as Address;
use Magento\ServiceProxy\Controller\Adminhtml\AbstractProxyController;

class Create extends AbstractProxyController implements HttpPostActionInterface, CsrfAwareActionInterface
{
    const ADMIN_RESOURCE = 'Magento_ServiceProxy::services';

    /**
     * @var OrderService
     */
    private $orderService;

    /**
     * @var Quote
     */
    private $quoteSession;

    /**
     * @param Context $context
     * @param Quote $quoteSession
     * @param OrderService $orderService
     */
    public function __construct(
        Context $context,
        Quote $quoteSession,
        OrderService $orderService
    ) {
        parent::__construct($context);
        $this->quoteSession = $quoteSession;
        $this->orderService = $orderService;
    }

    /**
     * @inheritDoc
     */
    public function execute(): ResultInterface
    {
        $result = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        try {
            $quote = $this->quoteSession->getQuote();
            $response = $this->orderService->create(
                [
                    'amount' => number_format($quote->getBaseGrandTotal(), 2, '.', ''),
                    'currency_code' => $quote->getCurrency()->getBaseCurrencyCode(),
                    'shipping_address' => $this->mapShippingAddress($quote->getShippingAddress()),
                    'is_digital' => $quote->isVirtual(),
                    'website_id' => $quote->getStore()->getWebsiteId()
                ]
            );
            $result->setHttpResponseCode($response['status'])
                ->setData(['response' => $response]);

        } catch (HttpException $e) {
            $result->setHttpResponseCode(500);
            $result->setData($e->getMessage());
        }
        return $result;
    }

    /**
     * Map shipping address.
     *
     * @param Address $shippingAddress
     * @return array|null
     */
    private function mapShippingAddress(Address $shippingAddress) :? array
    {
        return [
            'name' => $shippingAddress->getFirstname() . ' ' . $shippingAddress->getLastname(),
            'street' => [
                0 => $shippingAddress->getStreet()[0],
                1 => isset($shippingAddress->getStreet()[1]) ? $shippingAddress->getStreet()[1] : ''
            ],
            'region' => $shippingAddress->getRegion(),
            'city' => $shippingAddress->getCity(),
            'postcode' => $shippingAddress->getPostcode(),
            'country' => $shippingAddress->getCountry()
        ];
    }
}
