<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\PaymentServicesBase\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\ServicesConnector\Api\KeyValidationInterface;
use Magento\ServicesConnector\Exception\KeyNotFoundException;
use Magento\Store\Model\ScopeInterface;
use Magento\ServicesId\Model\ServicesConfig;
use Magento\Store\Model\StoreManagerInterface;

class Config
{
    private const EXTENSION_NAME = 'Magento_PaymentServicesBase';

    private const CONFIG_PATH_ENVIRONMENT = 'payment/payment_methods/method';

    private const CONFIG_PATH_MERCHANT_ID = 'payment/payment_methods/%s_merchant_id';

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var ServicesConfig
     */
    private $servicesConfig;

    /**
     * @var KeyValidationInterface
     */
    private $keyValidator;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param ServicesConfig $servicesConfig
     * @param KeyValidationInterface $keyValidator
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ServicesConfig $servicesConfig,
        KeyValidationInterface $keyValidator,
        StoreManagerInterface $storeManager
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->servicesConfig = $servicesConfig;
        $this->keyValidator = $keyValidator;
        $this->storeManager = $storeManager;
    }

    /**
     * Retrieve merchant ID from config
     *
     * @param string $environment
     * @return string
     */
    public function getMerchantId(string $environment = '') : string
    {
        $environment = $environment ?: $this->getEnvironmentType();
        return (string) $this->scopeConfig->getValue(
            $this->getMerchantIdConfigPath($environment),
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * Get SaaS environment ID
     *
     * @return null|string
     */
    public function getServicesEnvironmentId() : ?string
    {
        return $this->servicesConfig->getEnvironmentId();
    }

    /**
     * Get SaaS environment type
     *
     * @return string
     */
    public function getEnvironmentType() : string
    {
        return (string) $this->scopeConfig->getValue(
            self::CONFIG_PATH_ENVIRONMENT,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * Get environment types for all the websites for report display
     *
     * @return string
     */
    public function getEnvironmentTypeAcrossWebsites() : string
    {
        $environment = 'sandbox';
        $websites = $this->storeManager->getWebsites();
        foreach ($websites as $website) {
            $websiteEnvironment = (string) $this->scopeConfig->getValue(
                self::CONFIG_PATH_ENVIRONMENT,
                ScopeInterface::SCOPE_WEBSITE,
                $website->getId()
            );
            if ($websiteEnvironment === 'production') {
                $environment = $websiteEnvironment;
                break;
            }
        }
        return $environment;
    }

    /**
     * Get config path for merchant ID
     *
     * @param string $environment
     * @return string
     */
    private function getMerchantIdConfigPath(string $environment): string
    {
        return sprintf(self::CONFIG_PATH_MERCHANT_ID, $environment);
    }

    /**
     * Check is Magento Services configured.
     *
     * @param string $environment
     * @return bool
     */
    public function isMagentoServicesConfigured($environment = '') : bool
    {
        $environment = $environment ?: $this->getEnvironmentType();

        try {
            return $this->keyValidator->execute(self::EXTENSION_NAME, $environment) &&
                $this->servicesConfig->getEnvironmentId();
        } catch (KeyNotFoundException $exception) {
            return false;
        }
    }

    /**
     * Check if payments is enabled
     *
     * @return bool
     */
    public function isEnabled() : bool
    {
        return (bool) $this->scopeConfig->getValue(
            'payment/payment_services/active',
            ScopeInterface::SCOPE_STORE,
            null
        );
    }

    /**
     * Check if payments is configured
     *
     * @param string $environment
     * @return bool
     */
    public function isConfigured($environment = '') : bool
    {
        return $this->isMagentoServicesConfigured($environment)
            && $this->isEnabled()
            && $this->getMerchantId($environment);
    }
}
