<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\OutOfProcessTaxManagement\Plugin;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\CustomAttributeSerializable\Model\AttributesConfigurationPool;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Setup\Exception;
use Magento\Tax\Model\ClassModel;
use Magento\Tax\Model\ResourceModel\TaxClass\Collection as TaxClassCollection;
use Psr\Log\LoggerInterface;

class AddCustomAttributesAndSaveQuoteObject
{
    public function __construct(
        private readonly TaxClassCollection $taxClassCollection,
        private readonly ProductRepository  $productRepository,
        private readonly LoggerInterface    $logger
    ) {
    }

    public function beforeSave(CartRepositoryInterface $subject, CartInterface $quote): array
    {
        try {
            $customerTaxClass = $this->getQuoteCustomerTaxClass($quote);
            $customerTaxClassCustomAttributes = $this->getSerializedCustomAttributes($customerTaxClass);
            $this->applyCustomAttributes($quote, $customerTaxClassCustomAttributes);

            $this->setCustomAttributesOnQuoteItem($quote);
        } catch (Exception $e) {
            $this->logger->critical(
                'Error setting custom attributes on quote or items: ' . $e->getMessage(),
                ['exception' => $e]
            );
        }
        return [$quote];
    }

    private function setCustomAttributesOnQuoteItem(CartInterface $quote): void
    {
        foreach ($quote->getItems() ?? [] as $item) {
            $productId = $item->getProductId();
            if (!$productId) {
                continue;
            }
            try {
                $product = $this->productRepository->getById($productId);
            } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                $this->logger->warning("Product not found for id: {$productId}");
                continue;
            }

            $productTaxClass = $this->getTaxClass($product);
            $productTaxClassCustomAttributes = $this->getSerializedCustomAttributes($productTaxClass);
            $this->applyCustomAttributes($item, $productTaxClassCustomAttributes);
        }
    }

    private function applyCustomAttributes($entity, array $attributes): void
    {
        foreach ($attributes as $key => $value) {
            $currentAttribute = $entity->getCustomAttribute($key);
            $currentValue = $currentAttribute?->getValue();

            if ($currentValue !== $value) {
                $entity->setCustomAttribute($key, $value);
            }
        }
    }

    private function getTaxClass(ProductInterface $product): ?ClassModel
    {
        $taxClassId = $product->getTaxClassId();
        if (!$taxClassId) {
            return null;
        }

        $taxClass = $this->taxClassCollection->getItemById($taxClassId);
        return ($taxClass instanceof ClassModel) ? $taxClass : null;
    }

    private function getQuoteCustomerTaxClass(CartInterface $quote) : ?ClassModel
    {
        $taxClassId = $quote->getCustomerTaxClassId();
        if (!$taxClassId) {
            return null;
        }

        $taxClass = $this->taxClassCollection->getItemById(intval($taxClassId));
        return ($taxClass instanceof ClassModel) ? $taxClass : null;
    }

    private function getSerializedCustomAttributes(?\Magento\Framework\Model\AbstractModel $entity): array
    {
        $customAttributes = $entity?->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE);

        if (is_string($customAttributes)) {
            $customAttributes = json_decode($customAttributes, true);
        }

        if (!is_array($customAttributes)) {
            return [];
        }

        return $customAttributes;
    }
}
