<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\OutOfProcessTaxManagement\Test\Unit\Model;

use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\OutOfProcessTaxManagement\Api\Data\OutOfProcessTaxIntegrationInterface;
use Magento\OutOfProcessTaxManagement\Model\OutOfProcessTaxIntegration;
use Magento\OutOfProcessTaxManagement\Model\OutOfProcessTaxIntegrationFactory;
use Magento\OutOfProcessTaxManagement\Model\OutOfProcessTaxIntegrationRepository;
use Magento\OutOfProcessTaxManagement\Model\ResourceModel\OutOfProcessTaxIntegration as ResourceModel;
use Magento\OutOfProcessTaxManagement\Model\ResourceModel\OutOfProcessTaxIntegration\CollectionFactory;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class OutOfProcessTaxIntegrationRepositoryTest extends TestCase
{
    private OutOfProcessTaxIntegrationRepository $repository;
    private ResourceModel&MockObject $resourceModelMock;
    private OutOfProcessTaxIntegrationFactory&MockObject $taxIntegrationFactoryMock;
    private CollectionFactory&MockObject $collectionFactoryMock;

    protected function setUp(): void
    {
        $this->resourceModelMock = $this->createMock(ResourceModel::class);
        $this->taxIntegrationFactoryMock = $this->createMock(OutOfProcessTaxIntegrationFactory::class);
        $this->collectionFactoryMock = $this->createMock(CollectionFactory::class);

        $this->repository = new OutOfProcessTaxIntegrationRepository(
            $this->resourceModelMock,
            $this->taxIntegrationFactoryMock,
            $this->collectionFactoryMock
        );
    }

    public function testGetByCodeSuccess(): void
    {
        $taxIntegrationMock = $this->createMock(OutOfProcessTaxIntegration::class);
        $this->taxIntegrationFactoryMock->expects($this->once())
            ->method('create')
            ->willReturn($taxIntegrationMock);

        $this->resourceModelMock->expects($this->once())
            ->method('load')
            ->with($taxIntegrationMock, 'test_code', OutOfProcessTaxIntegrationInterface::FIELD_CODE)
            ->willReturnCallback(function ($object) {
                $object->setData(['code' => 'test_code']);
            });

        $taxIntegrationMock->expects($this->once())
            ->method('isObjectNew')
            ->willReturn(false);

        $result = $this->repository->getByCode('test_code');
        $this->assertSame($taxIntegrationMock, $result);
    }

    public function testGetByCodeNotFound(): void
    {
        $taxIntegrationMock = $this->createMock(OutOfProcessTaxIntegration::class);
        $this->taxIntegrationFactoryMock->expects($this->once())
            ->method('create')
            ->willReturn($taxIntegrationMock);

        $this->resourceModelMock->expects($this->once())
            ->method('load')
            ->with($taxIntegrationMock, 'invalid_code', OutOfProcessTaxIntegrationInterface::FIELD_CODE);

        $taxIntegrationMock->expects($this->once())
            ->method('isObjectNew')
            ->willReturn(true);

        $this->expectException(NoSuchEntityException::class);
        $this->repository->getByCode('invalid_code');
    }

    public function testGetList(): void
    {
        $collectionMock = $this->createMock(AbstractDb::class);
        $collectionMock->expects($this->once())
            ->method('getItems')
            ->willReturn(['item1', 'item2']);

        $this->collectionFactoryMock->expects($this->once())
            ->method('create')
            ->willReturn($collectionMock);

        $result = $this->repository->getList();
        $this->assertEquals(['item1', 'item2'], $result);
    }

    public function testCreateTaxIntegration(): void
    {
        $taxIntegrationMock = $this->createMock(OutOfProcessTaxIntegration::class);
        $taxIntegrationMock->expects($this->atLeastOnce())
            ->method('getCode')
            ->willReturn('test_code');
        $taxIntegrationMock->expects($this->exactly(2))
            ->method('isObjectNew')
            ->willReturnOnConsecutiveCalls(true, false);

        $this->taxIntegrationFactoryMock->expects($this->atLeastOnce())
            ->method('create')
            ->willReturn($taxIntegrationMock);

        $this->resourceModelMock->expects($this->once())
            ->method('save')
            ->with($taxIntegrationMock);

        $result = $this->repository->createOrUpdate($taxIntegrationMock);
        $this->assertSame($taxIntegrationMock, $result);
    }
}
