<?php
declare(strict_types=1);

namespace Magento\OutOfProcessTaxManagement\Model\Tax\Sales\Total\Quote;

use Magento\CustomAttributeSerializable\Model\AttributesConfigurationPool;
use Magento\Customer\Api\Data\AddressInterfaceFactory as CustomerAddressFactory;
use Magento\Customer\Api\Data\RegionInterfaceFactory as CustomerAddressRegionFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\OutOfProcessTaxManagement\Api\Data\OopAddressInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OopQuoteInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OopQuoteItemInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OopShippingInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OutOfProcessTaxIntegrationInterface;
use Magento\OutOfProcessTaxManagement\Api\OopTaxCalculationInterface;
use Magento\OutOfProcessTaxManagement\Api\OopTaxCollectionInterface;
use Magento\OutOfProcessTaxManagement\Api\OutOfProcessTaxIntegrationRepositoryInterface;
use Magento\OutOfProcessTaxManagement\Model\Data\OopAddress;
use Magento\OutOfProcessTaxManagement\Model\Data\OopQuote;
use Magento\OutOfProcessTaxManagement\Model\Data\OopQuoteItem;
use Magento\OutOfProcessTaxManagement\Model\Data\OopShipping;
use Magento\OutOfProcessTaxManagement\Model\Helper\RegionHelper;
use Magento\OutOfProcessTaxManagement\Model\Helper\ShippingOriginHelper;
use Magento\Quote\Api\Data\ShippingAssignmentInterface;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address\Total;
use Magento\Quote\Model\Quote\Address\Total\CollectorInterface;
use Magento\Tax\Api\Data\QuoteDetailsInterface;
use Magento\Tax\Api\Data\QuoteDetailsInterfaceFactory as QuoteDetailsFactory;
use Magento\Tax\Api\Data\QuoteDetailsItemInterface;
use Magento\Tax\Api\Data\QuoteDetailsItemInterfaceFactory as QuoteDetailsItemFactory;
use Magento\Tax\Api\Data\TaxClassKeyInterfaceFactory as TaxClassKeyFactory;
use Magento\Tax\Api\TaxCalculationInterface;
use Magento\Tax\Helper\Data;
use Magento\Tax\Model\ClassModel;
use Magento\Tax\Model\Config;
use Magento\Tax\Model\ResourceModel\TaxClass\Collection as TaxClassCollection;
use Psr\Log\LoggerInterface;

class Tax extends \Magento\Tax\Model\Sales\Total\Quote\Tax
{
    /**
     * Class constructor
     *
     * @param Config $taxConfig
     * @param TaxCalculationInterface $taxCalculationService
     * @param QuoteDetailsFactory $quoteDetailsFactory
     * @param QuoteDetailsItemFactory $quoteDetailsItemFactory
     * @param TaxClassKeyFactory $taxClassKeyFactory
     * @param CustomerAddressFactory $customerAddressFactory
     * @param CustomerAddressRegionFactory $customerAddressRegionFactory
     * @param Data $taxData
     * @param LoggerInterface $logger
     * @param OopTaxCollectionInterface $oopTaxCollector
     * @param OopTaxCalculationInterface $oopTaxCalculator
     * @param ShippingOriginHelper $shippingOriginHelper
     * @param OutOfProcessTaxIntegrationRepositoryInterface $oopTaxIntegrationRepository
     */
    public function __construct(
        Config                                               $taxConfig,
        TaxCalculationInterface                              $taxCalculationService,
        QuoteDetailsFactory                                            $quoteDetailsFactory,
        QuoteDetailsItemFactory                                        $quoteDetailsItemFactory,
        TaxClassKeyFactory                                             $taxClassKeyFactory,
        CustomerAddressFactory                                         $customerAddressFactory,
        CustomerAddressRegionFactory                                   $customerAddressRegionFactory,
        Data                                                           $taxData,
        private readonly LoggerInterface                               $logger,
        private readonly OopTaxCollectionInterface                     $oopTaxCollector,
        private readonly OopTaxCalculationInterface                    $oopTaxCalculator,
        private readonly ShippingOriginHelper                          $shippingOriginHelper,
        private readonly RegionHelper                                  $regionHelper,
        private readonly TaxClassCollection                            $taxClassCollection,
        private readonly OutOfProcessTaxIntegrationRepositoryInterface $oopTaxIntegrationRepository
    ) {
        parent::__construct(
            $taxConfig,
            $taxCalculationService,
            $quoteDetailsFactory,
            $quoteDetailsItemFactory,
            $taxClassKeyFactory,
            $customerAddressFactory,
            $customerAddressRegionFactory,
            $taxData
        );
    }

    /**
     * Collect tax totals for quote address
     *
     * @param Quote $quote
     * @param ShippingAssignmentInterface $shippingAssignment
     * @param Total $total
     * @return CollectorInterface
     */
    public function collect(
        Quote $quote,
        ShippingAssignmentInterface $shippingAssignment,
        Total $total
    ) {
        $oopTaxIntegration = $this->getActiveIntegration($quote);
        if (!$oopTaxIntegration) {
            return parent::collect($quote, $shippingAssignment, $total);
        }

        // copied from \Magento\Tax\Model\Sales\Total\Quote\Tax
        $this->clearValues($total);
        if (!$shippingAssignment->getItems()) {
            return $this;
        }

        // in core this is where the calculateTax "service" is called
        $baseQuoteDetails = $this->getQuoteDetails($shippingAssignment, $total, true);

        // init external quote details
        $baseOopQuote = $this->buildOopQuote($baseQuoteDetails, $quote);

        if (!$baseOopQuote->getShipToAddress()?->getPostcode()) {
            // If postcode is not present, then collect totals is being run from a context where customer has not submitted
            // their address, such as on the product listing, product detail, or cart page. Once the user enters their
            // postcode in the "Estimate Shipping & Tax" form on the cart page, or submits their shipping address in the
            // checkout, then a postcode will be present;
            return parent::collect($quote, $shippingAssignment, $total);
        }

        // collect applicable taxes - webhook extension point
        // taxes will be calculated on the base prices/amounts, not the current currency prices. The taxes for
        // the current currency will be calculated by multiplying the base tax rates * currency conversion rate.
        $baseOopQuote = $this->oopTaxCollector->collectTaxes($baseOopQuote);

        // calculate tax with base currency
        $baseTaxDetails = $this->oopTaxCalculator->calculateTaxes($baseQuoteDetails, $baseOopQuote);
        // calculate tax based on converted currencies
        $currencyRate = (float) ($quote->getCurrency()?->getBaseToQuoteRate() ?? 1.0);
        if ($currencyRate == 1.0) {
            $taxDetails = clone $baseTaxDetails;
        } else {
            $oopQuote = $this->oopTaxCalculator->calculateTaxCurrencies($baseOopQuote, $currencyRate);
            $quoteDetails = $this->getQuoteDetails($shippingAssignment, $total, false);
            $taxDetails = $this->oopTaxCalculator->calculateTaxes($quoteDetails, $oopQuote);
        }

        $itemsByType = $this->organizeItemTaxDetailsByType($taxDetails, $baseTaxDetails);
        if (isset($itemsByType[self::ITEM_TYPE_PRODUCT])) {
            $this->processProductItems($shippingAssignment, $itemsByType[self::ITEM_TYPE_PRODUCT], $total);
        }

        if (isset($itemsByType[self::ITEM_TYPE_SHIPPING])) {
            $shippingTaxDetails = $itemsByType[self::ITEM_TYPE_SHIPPING][self::ITEM_CODE_SHIPPING][self::KEY_ITEM];
            $baseShippingTaxDetails =
                $itemsByType[self::ITEM_TYPE_SHIPPING][self::ITEM_CODE_SHIPPING][self::KEY_BASE_ITEM];
            $this->processShippingTaxInfo($shippingAssignment, $total, $shippingTaxDetails, $baseShippingTaxDetails);
        }

        // Process taxable items that are not product or shipping
        $this->processExtraTaxables($total, $itemsByType);

        // Save applied taxes for each item and the quote in aggregation
        $this->processAppliedTaxes($total, $shippingAssignment, $itemsByType);

        return $this;
    }

    /**
     * @param Quote $quote
     * @return OutOfProcessTaxIntegrationInterface|bool
     * @throws NoSuchEntityException
     */
    private function getActiveIntegration(Quote $quote): OutOfProcessTaxIntegrationInterface|bool
    {
        foreach ($this->oopTaxIntegrationRepository->getList() as $taxIntegration) {
            if (!$taxIntegration->isActive()) {
                continue;
            }
            if (!$taxIntegration->getStores()) {
                return $taxIntegration;
            }
            if (in_array($quote->getStore()->getCode(), $taxIntegration->getStores())) {
                return $taxIntegration;
            }
        }
        return false;
    }

    /**
     * @param QuoteDetailsInterface $quoteDetails
     * @param Quote $quote
     * @return OopQuoteInterface
     */
    private function buildOopQuote(QuoteDetailsInterface $quoteDetails, Quote $quote): OopQuoteInterface
    {
        $oopQuote = new OopQuote();
        $oopQuoteItems = [];
        foreach ($quoteDetails->getItems() as $quoteItem) {
            /** @var ClassModel $taxClass */
            $taxClass = $this->getTaxClass($quoteItem);
            $oopQuoteItems[] = new OopQuoteItem([
                OopQuoteItemInterface::FIELD_CODE => $quoteItem->getCode(),
                OopQuoteItemInterface::FIELD_TYPE => $quoteItem->getType(),
                OopQuoteItemInterface::FIELD_UNIT_PRICE => $quoteItem->getUnitPrice(),
                OopQuoteItemInterface::FIELD_QUANTITY => $quoteItem->getQuantity(),
                OopQuoteItemInterface::FIELD_DISCOUNT_AMOUNT => $quoteItem->getDiscountAmount(),
                OopQuoteItemInterface::FIELD_IS_TAX_INCLUDED => $quoteItem->getIsTaxIncluded(),
                OopQuoteItemInterface::FIELD_TAX_CLASS => $taxClass?->getClassName(),
                OopQuoteItemInterface::FIELD_CUSTOM_ATTRIBUTES => $this->getSerializedCustomAttributes($taxClass),
                OopQuoteItemInterface::FIELD_SKU => $quoteItem->getExtensionAttributes()?->getProductSku(),
                OopQuoteItemInterface::FIELD_NAME => $quoteItem->getExtensionAttributes()?->getProductName(),
                // external system is expected to populate these fields
                OopQuoteItemInterface::FIELD_TAX => null,
                OopQuoteItemInterface::FIELD_TAX_BREAKDOWN => [],
            ]);
        }
        $oopQuote->setData(OopQuoteInterface::FIELD_ITEMS, $oopQuoteItems);
        $shippingOriginRegion = $this->shippingOriginHelper->getRegion();
        $oopQuote->setShipFromAddress(new OopAddress([
            OopAddressInterface::FIELD_STREET => $this->shippingOriginHelper->getStreet(),
            OopAddressInterface::FIELD_CITY => $this->shippingOriginHelper->getCity(),
            OopAddressInterface::FIELD_REGION => $shippingOriginRegion?->getName(),
            OopAddressInterface::FIELD_REGION_CODE => $shippingOriginRegion?->getCode(),
            OopAddressInterface::FIELD_COUNTRY => $this->shippingOriginHelper->getCountryId(),
            OopAddressInterface::FIELD_POSTCODE => $this->shippingOriginHelper->getPostcode(),
        ]));
        $shippingAddress = $quoteDetails->getShippingAddress();
        $shippingAddressRegion = $this->regionHelper->getRegion($shippingAddress);
        $oopQuote->setShipToAddress(new OopAddress([
            OopAddressInterface::FIELD_STREET => $shippingAddress?->getStreet(),
            OopAddressInterface::FIELD_CITY => $shippingAddress?->getCity(),
            OopAddressInterface::FIELD_REGION => $shippingAddressRegion?->getName(),
            OopAddressInterface::FIELD_REGION_CODE => $shippingAddressRegion?->getCode(),
            OopAddressInterface::FIELD_COUNTRY => $shippingAddress?->getCountryId(),
            OopAddressInterface::FIELD_POSTCODE => $shippingAddress?->getPostcode(),
        ]));
        $billingAddress = $quoteDetails->getBillingAddress();
        $billingAddressRegion = $this->regionHelper->getRegion($billingAddress);
        $oopQuote->setBillingAddress(new OopAddress([
            OopAddressInterface::FIELD_STREET => $billingAddress?->getStreet(),
            OopAddressInterface::FIELD_CITY => $billingAddress?->getCity(),
            OopAddressInterface::FIELD_REGION => $billingAddressRegion?->getName(),
            OopAddressInterface::FIELD_REGION_CODE => $billingAddressRegion?->getCode(),
            OopAddressInterface::FIELD_COUNTRY => $billingAddress?->getCountryId(),
            OopAddressInterface::FIELD_POSTCODE => $billingAddress?->getPostcode(),
        ]));
        $oopQuote->setShipping(new OopShipping([
            OopShippingInterface::FIELD_SHIPPING_METHOD => $quote->getShippingAddress()?->getShippingMethod(),
            OopShippingInterface::FIELD_SHIPPING_DESCRIPTION => $quote->getShippingAddress()?->getShippingDescription(),
        ]));

        /** @var ClassModel $taxClass */
        $customerTaxClass = $this->getCustomerTaxClass($quoteDetails);
        $oopQuote->setCustomerTaxClass($customerTaxClass?->getClassName());
        $oopQuote->setCustomAttributes($this->getSerializedCustomAttributes($customerTaxClass));

        return $oopQuote;
    }

    /**
     * Adapted from \Magento\Tax\Model\Sales\Total\Quote\Tax
     * Get quote details
     *
     * @param ShippingAssignmentInterface $shippingAssignment
     * @param Total $total
     * @param bool $useBaseCurrency
     * @return QuoteDetailsInterface
     */
    protected function getQuoteDetails(
        ShippingAssignmentInterface $shippingAssignment,
        Total $total,
        bool $useBaseCurrency
    ): QuoteDetailsInterface {
        $address = $shippingAssignment->getShipping()->getAddress();
        // Setup taxable items
        $priceIncludesTax = $this->_config->priceIncludesTax($address->getQuote()->getStore());
        /** @var QuoteDetailsItemInterface $itemDataObjects */
        $itemDataObjects = $this->mapItems($shippingAssignment, $priceIncludesTax, $useBaseCurrency);

        // Add shipping
        $shippingDataObject = $this->getShippingDataObject($shippingAssignment, $total, $useBaseCurrency);
        if ($shippingDataObject != null) {
            $itemDataObjects[] = $shippingDataObject;
        }

        // Process extra taxable items associated only with quote
        $quoteExtraTaxables = $this->mapQuoteExtraTaxables(
            $this->quoteDetailsItemDataObjectFactory,
            $address,
            $useBaseCurrency
        );
        if (!empty($quoteExtraTaxables)) {
            $itemDataObjects = array_merge($itemDataObjects, $quoteExtraTaxables);
        }

        return $this->prepareQuoteDetails($shippingAssignment, $itemDataObjects);
    }

    private function getTaxClass(QuoteDetailsItemInterface $quoteItem): ?ClassModel
    {
        $taxClassId = $quoteItem->getTaxClassId() ?? $quoteItem->getTaxClassKey()?->getValue();
        if (!$taxClassId) {
            return null;
        }

        $taxClass = $this->taxClassCollection->getItemById(intval($taxClassId));
        return ($taxClass instanceof ClassModel) ? $taxClass : null;
    }

    private function getCustomerTaxClass(QuoteDetailsInterface $quoteDetails) : ?ClassModel
    {
        $taxClassId = $quoteDetails->getCustomerTaxClassId() ?? $quoteDetails->getCustomerTaxClassKey()?->getValue();
        if (!$taxClassId) {
            return null;
        }

        $taxClass = $this->taxClassCollection->getItemById(intval($taxClassId));
        return ($taxClass instanceof ClassModel) ? $taxClass : null;
    }

    /**
     * Retrieves serialized custom attributes from a given entity (Quote or Quote\Item).
     *
     * @param \Magento\Framework\Model\AbstractModel|null $entity
     * @return array
     */
    private function getSerializedCustomAttributes(?\Magento\Framework\Model\AbstractModel $entity): array
    {
        $customAttributes = $entity?->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE);

        if (is_string($customAttributes)) {
            $customAttributes = json_decode($customAttributes, true);
        }

        if (!is_array($customAttributes)) {
            return [];
        }

        return $customAttributes;
    }
}
