<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\OutOfProcessTaxManagement\Model;

use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Phrase;
use Magento\OutOfProcessTaxManagement\Api\Data\OutOfProcessTaxIntegrationInterface;
use Magento\OutOfProcessTaxManagement\Api\OutOfProcessTaxIntegrationRepositoryInterface;
use Magento\OutOfProcessTaxManagement\Model\ResourceModel\OutOfProcessTaxIntegration as ResourceModel;
use Magento\OutOfProcessTaxManagement\Model\ResourceModel\OutOfProcessTaxIntegration\CollectionFactory;

class OutOfProcessTaxIntegrationRepository implements OutOfProcessTaxIntegrationRepositoryInterface
{
    /**
     * OutOfProcessTaxIntegrationRepository constructor.
     *
     * @param ResourceModel                     $resourceModel
     * @param OutOfProcessTaxIntegrationFactory $taxIntegrationFactory
     * @param CollectionFactory                 $collectionFactory
     */
    public function __construct(
        private readonly ResourceModel                     $resourceModel,
        private readonly OutOfProcessTaxIntegrationFactory $taxIntegrationFactory,
        private readonly CollectionFactory                 $collectionFactory,
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getByCode(string $code): OutOfProcessTaxIntegrationInterface
    {
        $taxIntegration = $this->findTaxIntegration($code);
        if ($taxIntegration === null) {
            throw new NoSuchEntityException(__('Out of process tax integration with code "%1" not found.', $code));
        }
        return $taxIntegration;
    }

    /**
     * @inheritDoc
     */
    public function getList(): array
    {
        $collection = $this->collectionFactory->create();
        return $collection->getItems();
    }

    /**
     * @inheritDoc
     */
    public function createOrUpdate(OutOfProcessTaxIntegrationInterface $tax_integration):
    OutOfProcessTaxIntegrationInterface
    {
        if ($tax_integration->isActive()) {
            $active = $this->findActiveIntegration();
            if ($active && $active->getCode() !== $tax_integration->getCode()) {
                throw new InvalidArgumentException(new Phrase(
                    "Only one active tax integration is supported: " .
                    "currently the tax integration with code '%activeCode%' is active, " .
                    "it has to be deactivated in order to activate '%currentCode%'.",
                    [
                        'activeCode' => $active->getCode(),
                        'currentCode' => $tax_integration->getCode(),
                    ]
                ));
            }
        }

        $existingTaxIntegration = $this->findTaxIntegration($tax_integration->getCode());
        if ($existingTaxIntegration) {
            $tax_integration->setId($existingTaxIntegration->getId());
        }
        $this->resourceModel->save($tax_integration);

        return $this->findTaxIntegration($tax_integration->getCode());
    }

    private function findTaxIntegration(string $code): ?OutOfProcessTaxIntegration
    {
        $taxIntegration = $this->taxIntegrationFactory->create();
        $this->resourceModel->load($taxIntegration, $code, OutOfProcessTaxIntegrationInterface::FIELD_CODE);
        return $taxIntegration->isObjectNew() ? null : $taxIntegration;
    }

    private function findActiveIntegration(): ?OutOfProcessTaxIntegration
    {
        $taxIntegration = $this->taxIntegrationFactory->create();
        $this->resourceModel->load($taxIntegration, '1', OutOfProcessTaxIntegrationInterface::FIELD_ACTIVE);
        return $taxIntegration->isObjectNew() ? null : $taxIntegration;
    }
}
