<?php
declare(strict_types=1);

namespace Magento\OutOfProcessTaxManagement\Model;

use Magento\Framework\Api\DataObjectHelper;
use Magento\OutOfProcessTaxManagement\Api\Data\OopQuoteInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OopQuoteItemInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OopQuoteItemTaxBreakdownInterface;
use Magento\OutOfProcessTaxManagement\Api\OopTaxCalculationInterface;
use Magento\Tax\Api\Data\AppliedTaxInterfaceFactory;
use Magento\Tax\Api\Data\AppliedTaxRateInterfaceFactory;
use Magento\Tax\Api\Data\QuoteDetailsInterface;
use Magento\Tax\Api\Data\QuoteDetailsItemInterface;
use Magento\Tax\Api\Data\TaxDetailsInterface;
use Magento\Tax\Api\Data\TaxDetailsInterfaceFactory;
use Magento\Tax\Api\Data\TaxDetailsItemInterface;
use Magento\Tax\Api\Data\TaxDetailsItemInterfaceFactory;
use Magento\Tax\Model\Calculation;
use Magento\Tax\Model\TaxDetails\AppliedTax;
use Magento\Tax\Model\TaxDetails\AppliedTaxRate;
use Magento\Tax\Model\TaxDetails\TaxDetails;
use Psr\Log\LoggerInterface;

/**
 * adapted from the \Magento\Tax\Model\TaxCalculation
 */
class OopTaxCalculation implements OopTaxCalculationInterface
{
    public function __construct(
        private readonly TaxDetailsInterfaceFactory     $taxDetailsDataObjectFactory,
        private readonly TaxDetailsItemInterfaceFactory $taxDetailsItemDataObjectFactory,
        private readonly AppliedTaxInterfaceFactory     $appliedTaxDataObjectFactory,
        private readonly AppliedTaxRateInterfaceFactory $appliedTaxRateDataObjectFactory,
        private readonly DataObjectHelper               $dataObjectHelper,
        private readonly Calculation                    $calculationTool,
        private readonly LoggerInterface                $logger
    ) {
    }

    /**
     * @inheritdoc
     */
    public function calculateTaxes(QuoteDetailsInterface $quoteDetails, OopQuoteInterface $oopQuote, int $storeId = null, bool $round = true): TaxDetailsInterface
    {
        // initial TaxDetails data
        $taxDetailsData = [
            TaxDetails::KEY_SUBTOTAL => 0.0,
            TaxDetails::KEY_TAX_AMOUNT => 0.0,
            TaxDetails::KEY_DISCOUNT_TAX_COMPENSATION_AMOUNT => 0.0,
            TaxDetails::KEY_APPLIED_TAXES => [],
            TaxDetails::KEY_ITEMS => [],
        ];

        $items = $quoteDetails->getItems();
        $oopItems = $oopQuote->getItems();
        if (empty($items) || empty($oopItems)) {
            return $this->taxDetailsDataObjectFactory->create()
                ->setSubtotal(0.0)
                ->setTaxAmount(0.0)
                ->setDiscountTaxCompensationAmount(0.0)
                ->setAppliedTaxes([])
                ->setItems([]);
        }

        $keyedItems = $this->getKeyedItems($items);
        $parentToChildren = $this->getParentToChildren($items);
        $keyedOopItems = $this->getKeyedOopItems($oopItems);

        $processedItems = [];
        /** @var QuoteDetailsItemInterface $item */
        foreach ($keyedItems as $item) {
            if (isset($parentToChildren[$item->getCode()])) {
                $processedChildren = [];
                foreach ($parentToChildren[$item->getCode()] as $child) {
                    $processedItem = $this->processItem($child, $keyedOopItems[$child->getCode()], $keyedItems);
                    $taxDetailsData = $this->aggregateItemData($taxDetailsData, $processedItem);
                    $processedItems[$processedItem->getCode()] = $processedItem;
                    $processedChildren[] = $processedItem;
                }
                $processedItem = $this->calculateParent($processedChildren, $item->getQuantity());
                $processedItem->setCode($item->getCode());
                $processedItem->setType($item->getType());
            } else {
                $processedItem = $this->processItem($item, $keyedOopItems[$item->getCode()], $keyedItems);
                $taxDetailsData = $this->aggregateItemData($taxDetailsData, $processedItem);
            }
            $processedItems[$processedItem->getCode()] = $processedItem;
        }

        $taxDetailsDataObject = $this->taxDetailsDataObjectFactory->create();
        $this->dataObjectHelper->populateWithArray(
            $taxDetailsDataObject,
            $taxDetailsData,
            TaxDetailsInterface::class
        );
        $taxDetailsDataObject->setItems($processedItems);

        return $taxDetailsDataObject;
    }

    /**
     * @inheritdoc
     */
    public function calculateTaxCurrencies(OopQuoteInterface $baseOopQuote, float $currencyRate): OopQuoteInterface
    {
        $oopQuote = clone $baseOopQuote;

        foreach ($oopQuote->getItems() as $item) {
            $tax = $item->getTax();
            if ($tax !== null) {
                $tax->setAmount($this->calculationTool->round($tax->getAmount() * $currencyRate));
                $tax->setDiscountCompensationAmount($this->calculationTool->round($tax->getDiscountCompensationAmount() * $currencyRate));
            }

            foreach ($item->getTaxBreakdown() as $taxBreakdown) {
                $taxBreakdown->setAmount($this->calculationTool->round($taxBreakdown->getAmount() * $currencyRate));
            }
        }

        return $oopQuote;
    }

    /**
     * Calculate item tax with customized rounding level
     *
     * @param QuoteDetailsItemInterface     $item
     * @param OopQuoteItemInterface         $oopItem
     * @param QuoteDetailsItemInterface[]   $keyedItems
     * @return TaxDetailsItemInterface
     */
    private function processItem(
        QuoteDetailsItemInterface $item,
        OopQuoteItemInterface     $oopItem,
        array                     $keyedItems
    ): TaxDetailsItemInterface {
        $quantity = $this->getTotalQuantity($item, $keyedItems);
        return $this->applyItemTaxes($item, $oopItem, $quantity);
    }

    /**
     * Add row total item amount to subtotal
     *
     * @param array $taxDetailsData
     * @param TaxDetailsItemInterface $item
     * @return array
     */
    protected function aggregateItemData(array $taxDetailsData, TaxDetailsItemInterface $item): array
    {
        $taxDetailsData[TaxDetails::KEY_SUBTOTAL]
            = $taxDetailsData[TaxDetails::KEY_SUBTOTAL] + $item->getRowTotal();

        $taxDetailsData[TaxDetails::KEY_TAX_AMOUNT]
            = $taxDetailsData[TaxDetails::KEY_TAX_AMOUNT] + $item->getRowTax();

        $taxDetailsData[TaxDetails::KEY_DISCOUNT_TAX_COMPENSATION_AMOUNT] =
            $taxDetailsData[TaxDetails::KEY_DISCOUNT_TAX_COMPENSATION_AMOUNT]
            + $item->getDiscountTaxCompensationAmount();

        $itemAppliedTaxes = $item->getAppliedTaxes();
        if ($itemAppliedTaxes === null) {
            return $taxDetailsData;
        }
        $appliedTaxes = $taxDetailsData[TaxDetails::KEY_APPLIED_TAXES];
        foreach ($itemAppliedTaxes as $taxId => $itemAppliedTax) {
            if (!isset($appliedTaxes[$taxId])) {
                //convert rate data object to array
                $rates = [];
                $rateDataObjects = $itemAppliedTax->getRates();
                foreach ($rateDataObjects as $rateDataObject) {
                    $rates[$rateDataObject->getCode()] = [
                        AppliedTaxRate::KEY_CODE => $rateDataObject->getCode(),
                        AppliedTaxRate::KEY_TITLE => $rateDataObject->getTitle(),
                        AppliedTaxRate::KEY_PERCENT => $rateDataObject->getPercent(),
                    ];
                }
                $appliedTaxes[$taxId] = [
                    AppliedTax::KEY_AMOUNT => $itemAppliedTax->getAmount(),
                    AppliedTax::KEY_PERCENT => $itemAppliedTax->getPercent(),
                    AppliedTax::KEY_RATES => $rates,
                    AppliedTax::KEY_TAX_RATE_KEY => $itemAppliedTax->getTaxRateKey(),
                ];
            } else {
                $appliedTaxes[$taxId][AppliedTax::KEY_AMOUNT] += $itemAppliedTax->getAmount();
            }
        }

        $taxDetailsData[TaxDetails::KEY_APPLIED_TAXES] = $appliedTaxes;
        return $taxDetailsData;
    }

    /**
     * Calculate row information for item based on children calculation
     *
     * @param TaxDetailsItemInterface[] $children
     * @param int $quantity
     * @return TaxDetailsItemInterface
     * @SuppressWarnings(PHPMD.UnusedLocalVariable)
     */
    private function calculateParent($children, $quantity): TaxDetailsItemInterface
    {
        $rowTotal = 0.00;
        $rowTotalInclTax = 0.00;
        $rowTax = 0.00;
        $taxableAmount = 0.00;

        foreach ($children as $child) {
            $rowTotal += $child->getRowTotal();
            $rowTotalInclTax += $child->getRowTotalInclTax();
            $rowTax += $child->getRowTax();
            $taxableAmount += $child->getTaxableAmount();
        }

        $price = $this->calculationTool->round($rowTotal / $quantity);
        $priceInclTax = $this->calculationTool->round($rowTotalInclTax / $quantity);

        return $this->taxDetailsItemDataObjectFactory->create()
            ->setPrice($price)
            ->setPriceInclTax($priceInclTax)
            ->setRowTotal($rowTotal)
            ->setRowTotalInclTax($rowTotalInclTax)
            ->setRowTax($rowTax);
    }

    /**
     * Calculates the total quantity for this item.
     *
     * What this really means is that if this is a child item, it return the parent quantity times
     * the child quantity and return that as the child's quantity.
     *
     * @param QuoteDetailsItemInterface $item
     * @param QuoteDetailsItemInterface[] $keyedItems
     * @return float
     */
    private function getTotalQuantity(QuoteDetailsItemInterface $item, array $keyedItems): float
    {
        if ($item->getParentCode()) {
            $parentQuantity = $keyedItems[$item->getParentCode()]->getQuantity();
            return $parentQuantity * $item->getQuantity();
        }
        return $item->getQuantity();
    }

    /**
     * @param QuoteDetailsItemInterface[] $items
     * @return array
     */
    private function getParentToChildren(array $items)
    {
        $parentToChildren = [];
        foreach ($items as $item) {
            if ($item->getParentCode() !== null) {
                $parentToChildren[$item->getParentCode()][] = $item;
            }
        }
        return $parentToChildren;
    }

    /**
     * @param QuoteDetailsItemInterface[] $items
     * @return array
     */
    private function getKeyedItems(array $items)
    {
        $keyedItems = [];
        foreach ($items as $item) {
            if ($item->getParentCode() === null) {
                $keyedItems[$item->getCode()] = $item;
            }
        }
        return $keyedItems;
    }

    /**
     * @param OopQuoteItemInterface[] $oopItems
     * @return array
     */
    private function getKeyedOopItems(array $oopItems)
    {
        $keyedOopItems = [];
        foreach ($oopItems as $oopItem) {
            $keyedOopItems[$oopItem->getCode()] = $oopItem;
        }
        return $keyedOopItems;
    }

    /**
     * @param QuoteDetailsItemInterface $item
     * @param OopQuoteItemInterface $oopItem
     * @param float $quantity
     * @return TaxDetailsItemInterface
     */
    private function applyItemTaxes(
        QuoteDetailsItemInterface $item,
        OopQuoteItemInterface     $oopItem,
        float                     $quantity
    ): TaxDetailsItemInterface {

        $taxRate = $oopItem->getTax()?->getRate() ?: 0.0;
        $rowTax = $oopItem->getTax()?->getAmount() ?: 0.0;
        $discountTaxCompensationAmount = $oopItem->getTax()?->getDiscountCompensationAmount() ?: 0.0;
        $uniTax = $quantity ? $this->calculationTool->round($rowTax / $quantity) : 0.0;

        if ($item->getIsTaxIncluded()) {
            // tax cannot be more than the total price
            $totalPriceInclTax = $item->getUnitPrice() * $quantity;
            $rowTax = min($totalPriceInclTax, $rowTax);
            $priceInclTax = $item->getUnitPrice();
            $uniTax = min($uniTax, $priceInclTax);

            $price = $priceInclTax - $uniTax;
            $rowTotalInclTax = $priceInclTax * $quantity;
            $rowTotal = $rowTotalInclTax - $rowTax;
        } else {
            $price = $item->getUnitPrice();
            $priceInclTax = $price + $uniTax;
            $rowTotal = $price * $quantity;
            $rowTotalInclTax = $rowTotal + $rowTax;
        }

        $appliedTaxes = $this->getAppliedTaxes($oopItem->getTaxBreakdown() ?? []);

        return $this->taxDetailsItemDataObjectFactory->create()
            ->setCode($item->getCode())
            ->setType($item->getType())
            ->setRowTax($rowTax)
            ->setPrice($price)
            ->setPriceInclTax($priceInclTax)
            ->setRowTotal($rowTotal)
            ->setRowTotalInclTax($rowTotalInclTax)
            ->setTaxPercent($taxRate)
            ->setDiscountTaxCompensationAmount($discountTaxCompensationAmount)
            ->setAssociatedItemCode($item->getAssociatedItemCode())
            ->setAppliedTaxes($appliedTaxes);
    }

    /**
     * @param OopQuoteItemTaxBreakdownInterface[] $taxBreakdown
     * @return array
     */
    private function getAppliedTaxes(array $taxBreakdown): array
    {
        return array_map(fn ($tax) => $this->appliedTaxDataObjectFactory->create()
            ->setAmount($tax->getAmount())
            ->setPercent($tax->getRate())
            ->setTaxRateKey($tax->getTaxRateKey())
            ->setRates([
                $this->appliedTaxRateDataObjectFactory->create()
                    ->setCode($tax->getCode())
                    ->setPercent($tax->getRate())
                    ->setTitle($tax->getTitle())
            ]), $taxBreakdown);
    }
}
