<?php

declare(strict_types=1);

use Magento\Framework\Interception\PluginListInterface;
use Magento\Framework\ObjectManagerInterface;
use Magento\OutOfProcessTaxManagement\Api\Data\OutOfProcessTaxIntegrationInterfaceFactory;
use Magento\OutOfProcessTaxManagement\Model\OutOfProcessTaxIntegrationRepository;
use Magento\Quote\Api\CartItemRepositoryInterface;
use Magento\Quote\Model\Quote;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Api\Data\ProductInterfaceFactory;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\CartManagementInterface;
use Magento\Quote\Api\Data\CartItemInterfaceFactory;
use Magento\Store\Model\StoreManagerInterface;
use PHPUnit\Framework\TestCase;

class OopTaxCalculationTest extends TestCase
{
    private ?ObjectManagerInterface $objectManager;
    private ?ProductRepositoryInterface $productRepository;
    private ?ProductInterfaceFactory $productFactory;
    private ?QuoteFactory $quoteFactory;
    private ?CartRepositoryInterface $cartRepository;
    private ?CartManagementInterface $cartManagement;
    private ?CartItemInterfaceFactory $cartItemFactory;
    private ?StoreManagerInterface $storeManager;
    private ?CartItemRepositoryInterface $cartItemRepository;
    private ?OutOfProcessTaxIntegrationRepository $oopTaxIntegrationRepository;
    private ?OutOfProcessTaxIntegrationInterfaceFactory $oopTaxIntegrationFactory = null;
    private ?PluginListInterface $pluginList;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->productRepository = $this->objectManager->get(ProductRepositoryInterface::class);
        $this->productFactory = $this->objectManager->get(ProductInterfaceFactory::class);
        $this->quoteFactory = $this->objectManager->get(QuoteFactory::class);
        $this->cartRepository = $this->objectManager->get(CartRepositoryInterface::class);
        $this->cartManagement = $this->objectManager->get(CartManagementInterface::class);
        $this->cartItemFactory = $this->objectManager->get(CartItemInterfaceFactory::class);
        $this->storeManager = $this->objectManager->get(StoreManagerInterface::class);
        $this->cartItemRepository = $this->objectManager->get(CartItemRepositoryInterface::class);
        $this->oopTaxIntegrationRepository = $this->objectManager->get(OutOfProcessTaxIntegrationRepository::class);
        $this->oopTaxIntegrationFactory = $this->objectManager->get(OutOfProcessTaxIntegrationInterfaceFactory::class);
        $this->pluginList = $this->objectManager->get(PluginListInterface::class);
    }

    public function testCreateQuoteAndAddProduct()
    {
        $product = $this->createProduct();

        //TODO: Try to create the quote using GraphQL instead of use repository
        $quote = $this->createQuote();
        $cartItem = $this->createQuoteItem($product, $quote);

        $quote->setItems([$cartItem]);
        $this->setQuoteShippingMethod($quote);
        $this->createOopTaxIntegration();

        // Save the Quote implicitly calls OopTaxCalculation::calculateTaxes which will execute the webhook
        $this->cartRepository->save($quote);

        //Assert that taxes are applied to the cart
        $cart = $this->cartRepository->get($quote->getId());
        //Check item final price without taxes
        $this->assertEquals(10, $cart->getItems()[0]->getPrice());
        //Check item final price with taxes
        $this->assertEquals(10.5, $cart->getItems()[0]->getPriceInclTax());
        //Check shipping final price without taxes
        $this->assertEquals(5, $cart->getShippingAddress()->getShippingAmount());
        //Check shipping final price with taxes
        $this->assertEquals(5.25, $cart->getShippingAddress()->getShippingInclTax());
        //Check cart final price with taxes
        $this->assertEquals(15.75, $cart->getGrandTotal());
    }

    /**
     * @param Quote $quote
     * @return void
     */
    private function setQuoteShippingMethod(Quote $quote): void
    {
        $quote->getShippingAddress()
            ->setCountryId('US')
            ->setPostcode('10001')
            ->setCity('New York')
            ->setRegion('NY')
            ->setRegionId(43)
            ->setStreet(['123 Test Street'])
            ->setTelephone('1234567890')
            ->setShippingMethod('flatrate_flatrate')
            ->setCollectShippingRates(true);
    }

    /**
     * @return \Magento\Catalog\Api\Data\ProductInterface
     * @throws \Magento\Framework\Exception\CouldNotSaveException
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\StateException
     */
    private function createProduct(): \Magento\Catalog\Api\Data\ProductInterface
    {
        $product = $this->productFactory->create();
        $product->setSku('test-product')
            ->setName('Test Product')
            ->setPrice(10.00)
            ->setTypeId(\Magento\Catalog\Model\Product\Type::TYPE_SIMPLE)
            ->setAttributeSetId(4) // Default attribute set
            ->setVisibility(\Magento\Catalog\Model\Product\Visibility::VISIBILITY_BOTH)
            ->setStatus(\Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED)
            ->setStockData(['qty' => 100, 'is_in_stock' => 1]);
        $this->productRepository->save($product);
        return $product; // Ensure product is saved
    }

    /**
     * @return Quote
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function createQuote(): Quote
    {
        $quote = $this->quoteFactory->create();
        $quote->setStore($this->storeManager->getStore());
        $quote->setIsActive(true);
        return $quote;
    }

    /**
     * @param \Magento\Catalog\Api\Data\ProductInterface $product
     * @param Quote $quote
     * @return mixed
     */
    private function createQuoteItem(\Magento\Catalog\Api\Data\ProductInterface $product, Quote $quote)
    {
        $cartItem = $this->cartItemFactory->create();
        $cartItem
            ->setSku($product->getSku())
            ->setProductType("simple")
            ->setQuoteId($quote->getId())
            ->setProduct($product)
            ->setQty(1)
            ->setPrice($product->getPrice());
        return $cartItem;
    }

    /**
     * @return void
     */
    private function createOopTaxIntegration(): void
    {
        $oopTaxIntegration = $this->oopTaxIntegrationFactory->create();
        $oopTaxIntegration->setCode('test-code')
            ->setTitle('Test Code')
            ->setActive(true);
        $this->oopTaxIntegrationRepository->createOrUpdate($oopTaxIntegration);
    }
}
