<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Plugin\Payment\Block\Form;

use Magento\AdobeCommerceWebhooks\Model\Config\WebhookConfigurationException;
use Magento\AdobeCommerceWebhooks\Model\WebhookList;
use Magento\OutOfProcessPaymentMethods\Api\PaymentMethodFilterInterface;
use Magento\OutOfProcessPaymentMethods\Model\PaymentMethodFilter\PayloadFactory;
use Magento\Payment\Block\Form\Container;

/**
 * Plugin to filter payment methods in admin order creation
 */
class ContainerAfterGetMethodsPlugin
{
    private const WEBHOOK_NAME = 'plugin.magento.out_of_process_payment_methods.api.payment_method_filter.get_list';
    private const WEBHOOK_TYPE = 'after';

    /**
     * @param PayloadFactory $payloadFactory
     * @param WebhookList $webhookList
     * @param PaymentMethodFilterInterface $paymentMethodFilter
     */
    public function __construct(
        private readonly PayloadFactory $payloadFactory,
        private readonly WebhookList $webhookList,
        private readonly PaymentMethodFilterInterface $paymentMethodFilter
    ) {
    }

    /**
     * Intercepts getMethods method and filter payment methods based on the webhook response.
     *
     * Check if the appropriate webhook is registered and if not return the original result without any
     * additional processing to avoid unnecessary overhead.
     *
     * @param Container $subject
     * @param array $result
     * @return array
     */
    public function afterGetMethods(Container $subject, array $result): array
    {
        if (!$this->checkIfWebhookIsEnabled()) {
            return $result;
        }

        if (empty($result)) {
            return $result;
        }

        $quote = $subject->getQuote();
        $cartId = $quote ? $quote->getId() : null;

        if ($cartId === null) {
            return $result;
        }

        $payload = $this->payloadFactory->create($cartId, $result);

        $methodsToFilter = array_column(
            $this->paymentMethodFilter->getList($payload),
            'code'
        );

        if (empty($methodsToFilter)) {
            return $result;
        }

        foreach ($result as $key => $paymentMethod) {
            if (in_array($paymentMethod->getCode(), $methodsToFilter)) {
                unset($result[$key]);
            }
        }

        $subject->setData('methods', $result);
        return $result;
    }

    /**
     * Check if the webhook for filtering payment methods is enabled.
     *
     * @return bool
     */
    private function checkIfWebhookIsEnabled(): bool
    {
        try {
            if ($this->webhookList->get(self::WEBHOOK_NAME, self::WEBHOOK_TYPE)) {
                return true;
            }
        } catch (WebhookConfigurationException $e) {
        }

        return false;
    }
}
