<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Model;

use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use Magento\OutOfProcessPaymentMethods\Api\Data\OutOfProcessPaymentMethodInterface;
use Magento\OutOfProcessPaymentMethods\Api\PaymentMethodAdapterInterface;
use Magento\Payment\Model\InfoInterface;
use Magento\Quote\Api\Data\CartInterface;

class PaymentMethodAdapter implements PaymentMethodAdapterInterface
{
    /**
     * @var int $store
     */
    private $store;

    /**
     * @var InfoInterface $infoInstance
     */
    private $infoInstance;

    /**
     * @param OutOfProcessPaymentMethod $outOfProcessPaymentMethod
     */
    public function __construct(private readonly OutOfProcessPaymentMethod $outOfProcessPaymentMethod)
    {
    }

    /**
     * @inheritDoc
     */
    public function getCode()
    {
        return $this->outOfProcessPaymentMethod->getCode();
    }

    /**
     * @inheritDoc
     */
    public function getFormBlockType()
    {
        return \Magento\Payment\Block\Form::class;
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        return $this->outOfProcessPaymentMethod->getTitle();
    }

    /**
     * @inheritDoc
     */
    public function setStore($storeId)
    {
        $this->store = (int)$storeId;
    }

    /**
     * @inheritDoc
     */
    public function getStore()
    {
        return $this->store;
    }

    /**
     * @inheritDoc
     */
    public function canOrder()
    {
        return $this->getCustomConfigValue('can_order') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canAuthorize()
    {
        return $this->getCustomConfigValue('can_authorize') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canCapture()
    {
        return $this->getCustomConfigValue('can_capture') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canCapturePartial()
    {
        return $this->getCustomConfigValue('can_capture_partial') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function canCaptureOnce()
    {
        return $this->getCustomConfigValue('can_capture_once') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canRefund()
    {
        return $this->getCustomConfigValue('can_refund') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canRefundPartialPerInvoice()
    {
        return $this->getCustomConfigValue('can_refund_partial_per_invoice') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function canVoid()
    {
        return $this->getCustomConfigValue('can_void') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canUseInternal()
    {
        return $this->getCustomConfigValue('can_use_internal') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canUseCheckout()
    {
        return $this->getCustomConfigValue('can_use_checkout') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function canEdit()
    {
        return $this->getCustomConfigValue('can_edit') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function canFetchTransactionInfo()
    {
        return $this->getCustomConfigValue('can_fetch_transaction_info') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function fetchTransactionInfo(InfoInterface $payment, $transactionId)
    {
        return [];
    }

    /**
     * @inheritDoc
     */
    public function isGateway()
    {
        return $this->getCustomConfigValue('is_gateway') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function isOffline()
    {
        return $this->getCustomConfigValue('is_offline') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function isInitializeNeeded()
    {
        return $this->getCustomConfigValue('is_initialization_needed') ?? false;
    }

    /**
     * @inheritDoc
     */
    public function canUseForCountry($country)
    {
        $countries = $this->outOfProcessPaymentMethod->getCountries();
        return empty($countries) || in_array($country, $countries);
    }

    /**
     * @inheritDoc
     */
    public function canUseForCurrency($currencyCode)
    {
        $currencies = $this->outOfProcessPaymentMethod->getCurrencies();
        return empty($currencies) || in_array($currencyCode, $currencies);
    }

    /**
     * @inheritDoc
     */
    public function getInfoBlockType()
    {
        return \Magento\Payment\Block\Info::class;
    }

    /**
     * @inheritDoc
     */
    public function getInfoInstance()
    {
        $instance = $this->infoInstance;
        if (!$instance instanceof InfoInterface) {
            throw new LocalizedException(
                __('We cannot retrieve the payment information object instance.')
            );
        }
        return $instance;
    }

    /**
     * @inheritDoc
     */
    public function setInfoInstance(InfoInterface $info)
    {
        $this->infoInstance = $info;
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        // TODO - pending addition of directory
        //        $paymentInfo = $this->getInfoInstance();
        //        if ($paymentInfo instanceof Payment) {
        //            $billingCountry = $paymentInfo->getOrder()->getBillingAddress()->getCountryId();
        //        } else {
        //            $billingCountry = $paymentInfo->getQuote()->getBillingAddress()->getCountryId();
        //        }
        //        $billingCountry = $billingCountry ?: $this->directory->getDefaultCountry();
        //
        //        if (!$this->canUseForCountry($billingCountry)) {
        //            throw new \Magento\Framework\Exception\LocalizedException(
        //                __('You can\'t use the payment type you selected to make payments to the billing country.')
        //            );
        //        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function order(InfoInterface $payment, $amount)
    {
        if (!$this->canOrder()) {
            throw new LocalizedException(__('The order action is not available.'));
        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function authorize(InfoInterface $payment, $amount)
    {
        if (!$this->canAuthorize()) {
            throw new LocalizedException(__('The authorize action is not available.'));
        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function capture(InfoInterface $payment, $amount)
    {
        if (!$this->canCapture()) {
            throw new LocalizedException(__('The capture action is not available.'));
        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function refund(InfoInterface $payment, $amount)
    {
        if (!$this->canRefund()) {
            throw new LocalizedException(__('The refund action is not available.'));
        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function cancel(InfoInterface $payment)
    {
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function void(InfoInterface $payment)
    {
        if (!$this->canVoid()) {
            throw new LocalizedException(__('The void action is not available.'));
        }
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function canReviewPayment()
    {
        return $this->getCustomConfigValue('can_review_payment') ?? true;
    }

    /**
     * @inheritDoc
     */
    public function acceptPayment(InfoInterface $payment)
    {
        if (!$this->canReviewPayment()) {
            throw new LocalizedException(__('The payment review action is unavailable.'));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function denyPayment(InfoInterface $payment)
    {
        if (!$this->canReviewPayment()) {
            throw new LocalizedException(__('The payment review action is unavailable.'));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function getConfigData($field, $storeId = null)
    {
        // TODO - add store context
        if ($field == OutOfProcessPaymentMethodInterface::FIELD_ORDER_STATUS) {
            return $this->outOfProcessPaymentMethod->getOrderStatus();
        }
        if ($field == OutOfProcessPaymentMethodInterface::FIELD_TITLE) {
            return $this->outOfProcessPaymentMethod->getTitle();
        }
        return $this->getCustomConfigValue($field);
    }

    /**
     * @inheritDoc
     */
    public function assignData(DataObject $data)
    {
        // TODO - figure this out
        //        $this->_eventManager->dispatch(
        //            'payment_method_assign_data_' . $this->getCode(),
        //            [
        //                AbstractDataAssignObserver::METHOD_CODE => $this,
        //                AbstractDataAssignObserver::MODEL_CODE => $this->getInfoInstance(),
        //                AbstractDataAssignObserver::DATA_CODE => $data
        //            ]
        //        );
        //
        //        $this->_eventManager->dispatch(
        //            'payment_method_assign_data',
        //            [
        //                AbstractDataAssignObserver::METHOD_CODE => $this,
        //                AbstractDataAssignObserver::MODEL_CODE => $this->getInfoInstance(),
        //                AbstractDataAssignObserver::DATA_CODE => $data
        //            ]
        //        );
        //
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function isAvailable(?CartInterface $quote = null)
    {
        return $this->isActive($quote?->getStoreId());
    }

    /**
     * @inheritDoc
     */
    public function isActive($storeId = null)
    {
        if (!$this->outOfProcessPaymentMethod->isActive()) {
            return false;
        }

        $stores = $this->outOfProcessPaymentMethod->getStoreIds();
        return empty($stores) || in_array($storeId, $stores);
    }

    /**
     * @inheritDoc
     */
    public function initialize($paymentAction, $stateObject)
    {
        return $this;
    }

    /**
     * @inheritDoc
     */
    public function getConfigPaymentAction()
    {
        // out of process offline payments do not support payment actions
        return null;
    }

    /**
     * Get the value of a key from the custom config
     *
     * @param string $key The key to retrieve from the custom config
     * @return string|null The value of the key in the custom config or null if not found
     */
    private function getCustomConfigValue($key): ?string
    {
        foreach ($this->outOfProcessPaymentMethod->getCustomConfig() as $kv) {
            if ($kv->getKey() === $key) {
                return $kv->getValue();
            }
        }
        return null;
    }
}
