<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Model\PaymentMethodFilter;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Customer\Model\Session;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\OutOfProcessPaymentMethods\Api\Data\PaymentMethodFilterPayloadInterface;
use Magento\OutOfProcessPaymentMethods\Api\Data\PaymentMethodFilterPayloadInterfaceFactory;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Api\Data\CartItemInterface;
use Magento\Quote\Api\Data\PaymentMethodInterface;

/**
 * Creates payload for payment method filter request
 */
class PayloadFactory
{
    private const FIELD_ATTRIBUTES = 'attributes';
    private const FIELD_PAYMENT_METHOD_CODE = 'code';
    private const FIELD_PAYMENT_METHOD_TITLE = 'title';

    /**
     * @param CartRepositoryInterface $cartRepository
     * @param PaymentMethodFilterPayloadInterfaceFactory $payloadInterfaceFactory
     * @param ProductRepositoryInterface $productRepository
     * @param Session $session
     */
    public function __construct(
        private readonly CartRepositoryInterface $cartRepository,
        private readonly PaymentMethodFilterPayloadInterfaceFactory $payloadInterfaceFactory,
        private readonly ProductRepositoryInterface $productRepository,
        private readonly Session $session
    ) {
    }

    /**
     * Creates payload for payment method filter request
     *
     * @param $cartId
     * @param PaymentMethodInterface[] $paymentMethods
     * @return PaymentMethodFilterPayloadInterface
     */
    public function create($cartId, array $paymentMethods = []): PaymentMethodFilterPayloadInterface
    {
        try {
            $cart = $this->cartRepository->get($cartId);
            $this->addProductCustomAttributes($cart);
        } catch (NoSuchEntityException $e) {
            $cart = null;
        }

        return $this->payloadInterfaceFactory->create([
            'data' => [
                PaymentMethodFilterPayloadInterface::FIELD_CART => $cart,
                PaymentMethodFilterPayloadInterface::FIELD_CUSTOMER => $this->session->isLoggedIn() ?
                    $this->session->getCustomer() : null,
                PaymentMethodFilterPayloadInterface::FIELD_AVAILABLE_PAYMENT_METHODS => array_map(
                    fn ($paymentMethod) => [
                        self::FIELD_PAYMENT_METHOD_CODE => $paymentMethod?->getCode() ?? '',
                        self::FIELD_PAYMENT_METHOD_TITLE => $paymentMethod?->getTitle() ?? '',
                    ],
                    $paymentMethods
                )
            ]
        ]);
    }

    /**
     * Add custom attributes to products in cart
     *
     * @param CartInterface $cart
     */
    private function addProductCustomAttributes(CartInterface $cart): void
    {
        foreach ($cart->getItems() as $item) {
            $item->getProduct()->setData(
                self::FIELD_ATTRIBUTES,
                $this->getProductAttributes($item)
            );
        }
    }

    /**
     * Returns the list of custom attributes for product from quote item
     *
     * @param CartItemInterface $item
     * @return array
     */
    private function getProductAttributes(CartItemInterface $item): array
    {
        $customAttributes = [];

        try {
            $product = $this->productRepository->get($item->getSku());
            $attributes = $product->getAttributes();

            foreach ($product->getCustomAttributes() as $customAttribute) {
                if (!isset($attributes[$customAttribute->getAttributeCode()])) {
                    continue;
                }
                $attribute = $attributes[$customAttribute->getAttributeCode()];
                $customAttributes[$attribute->getAttributeCode()] = $attribute->getFrontend()->getValue($product);
            }
        } catch (NoSuchEntityException $e) {
            return [];
        }

        return $customAttributes;
    }
}
