<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Model;

use InvalidArgumentException;
use Magento\Framework\App\Cache\Manager as CacheManager;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\OutOfProcessPaymentMethods\Api\Data\OutOfProcessPaymentMethodInterface;
use Magento\OutOfProcessPaymentMethods\Model\Helper\Store;
use Magento\OutOfProcessPaymentMethods\Model\ResourceModel\OutOfProcessPaymentMethod as ResourceModel;
use Magento\Payment\Helper\Data as PaymentHelper;

class OutOfProcessPaymentMethod extends AbstractModel implements OutOfProcessPaymentMethodInterface
{
    protected $_eventPrefix = 'oope_payment_method';

    /**
     * @param Store $storeHelper
     * @param WriterInterface $configWriter
     * @param CacheManager $cacheManager
     * @param Context $context
     * @param Registry $registry
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        private readonly Store           $storeHelper,
        private readonly WriterInterface $configWriter,
        private readonly CacheManager    $cacheManager,
        Context                          $context,
        Registry                         $registry,
        ?AbstractResource                $resource = null,
        ?AbstractDb                      $resourceCollection = null,
        array                            $data = []
    ) {
        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
    }

    /**
     * @inheritDoc
     */
    protected function _construct(): void
    {
        $this->_init(ResourceModel::class);
    }

    /**
     * @inheritDoc
     */
    public function getCode(): string
    {
        return $this->getData(self::FIELD_CODE);
    }

    /**
     * @inheritDoc
     */
    public function setCode(string $code): self
    {
        return $this->setData(self::FIELD_CODE, $code);
    }

    /**
     * @inheritDoc
     */
    public function getTitle(): string
    {
        return $this->getData(self::FIELD_TITLE);
    }

    /**
     * @inheritDoc
     */
    public function setTitle(string $title): self
    {
        return $this->setData(self::FIELD_TITLE, $title);
    }

    /**
     * @inheritDoc
     */
    public function isActive(): bool
    {
        return $this->getData(self::FIELD_ACTIVE) === '1';
    }

    /**
     * @inheritDoc
     */
    public function setActive(bool $active): self
    {
        return $this->setData(self::FIELD_ACTIVE, $active);
    }

    /**
     * @inheritDoc
     */
    public function getBackendIntegrationUrl(): string
    {
        return $this->getData(self::FIELD_BACKEND_INTEGRATION_URL);
    }

    /**
     * @inheritDoc
     */
    public function setBackendIntegrationUrl(string $backendIntegrationUrl): self
    {
        return $this->setData(self::FIELD_BACKEND_INTEGRATION_URL, $backendIntegrationUrl);
    }

    /**
     * @inheritDoc
     */
    public function setStores(array $stores): self
    {
        $stores = $this->storeHelper->getByCode($stores);
        return $this->setData(self::FIELD_STORE_IDS, array_map(fn ($store) => $store->getId(), $stores));
    }

    /**
     * @inheritDoc
     */
    public function getStores(): array
    {
        $stores = $this->storeHelper->getById($this->getData(self::FIELD_STORE_IDS));
        return array_map(fn ($store) => $store->getCode(), $stores);
    }

    /**
     * @inheritDoc
     */
    public function getStoreIds(): array
    {
        return $this->getData(self::FIELD_STORE_IDS);
    }

    /**
     * @inheritDoc
     */
    public function getOrderStatus(): ?string
    {
        return $this->getData(self::FIELD_ORDER_STATUS);
    }

    /**
     * @inheritDoc
     */
    public function setOrderStatus(?string $orderStatus): self
    {
        return $this->setData(self::FIELD_ORDER_STATUS, $orderStatus);
    }

    /**
     * @inheritDoc
     */
    public function getCountries(): array
    {
        return $this->getData(self::FIELD_COUNTRIES) ?? [];
    }

    /**
     * @inheritDoc
     */
    public function setCountries(?array $countries): self
    {
        foreach ($countries as $country) {
            if (!is_string($country) || strlen($country) != 2) {
                throw new InvalidArgumentException(
                    "\"$country\" has invalid country format: should be a string of 2 letters"
                );
            }
        }
        return $this->setData(self::FIELD_COUNTRIES, $countries);
    }

    /**
     * @inheritDoc
     */
    public function getCurrencies(): array
    {
        return $this->getData(self::FIELD_CURRENCIES) ?? [];
    }

    /**
     * @inheritDoc
     */
    public function setCurrencies(?array $currencies): self
    {
        foreach ($currencies as $currency) {
            if (!is_string($currency) || strlen($currency) != 3) {
                throw new InvalidArgumentException(
                    "\"$currency\" has invalid currency format: should be a string of 3 letters"
                );
            }
        }
        return $this->setData(self::FIELD_CURRENCIES, $currencies);
    }

    /**
     * @inheritDoc
     */
    public function getCustomConfig(): array
    {
        return $this->getData(self::FIELD_CUSTOM_CONFIG) ?? [];
    }

    /**
     * @inheritDoc
     */
    public function setCustomConfig(array $customConfig): self
    {
        return $this->setData(self::FIELD_CUSTOM_CONFIG, $customConfig);
    }

    /**
     * Update the default payment method title in the core configuration.
     * Note config cache is cleared if the title has changed.
     *
     * @return $this
     * @see \Magento\Payment\Helper\Data::getPaymentMethodList
     */
    public function afterSave(): self
    {
        $originalTitle = $this->getOrigData(self::FIELD_TITLE);
        $currentTitle = $this->getTitle();

        if ($originalTitle != $currentTitle) {
            // Save the payment method title in core_config_data
            $configPath = sprintf('%s/%s/title', PaymentHelper::XML_PATH_PAYMENT_METHODS, $this->getCode());
            $this->configWriter->save($configPath, $this->getTitle(), ScopeConfigInterface::SCOPE_TYPE_DEFAULT);

            // Clear the config cache
            $cacheTypes = $this->cacheManager->getAvailableTypes();
            if (in_array('config', $cacheTypes, true)) {
                $this->cacheManager->clean(['config']);
            }
        }

        return parent::afterSave();
    }
}
