<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Model\Plugin;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\OrderRepositoryInterface;

class GetOrdersByCartPlugin
{

    /**
     * @param ResourceConnection $resourceConnection
     */
    public function __construct(private readonly ResourceConnection $resourceConnection)
    {
    }

    /**
     * Intercepts query list operation to replace quote masked id (aka cart id) filter to the corresponding filter by
     * numeric quote id.
     * All the other filters are not mutated so its original values are kept.
     *
     * @param OrderRepositoryInterface $subject intercepted class
     * @param SearchCriteriaInterface $searchCriteria the original search criteria
     * @return SearchCriteriaInterface[] when masked_quote_id is found the mutated search criteria, the original search
     * criteria otherwise
     * @throws NoSuchEntityException when masked_quote_id is not found
     */
    public function beforeGetList(OrderRepositoryInterface $subject, SearchCriteriaInterface $searchCriteria): array
    {
        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                if ($filter->getField() === 'masked_quote_id') {
                    $quoteId = $this->getQuoteFromMasked($filter->getValue());
                    $filter->setField('quote_id');
                    $filter->setValue($quoteId);
                    $filter->setConditionType('eq');
                }
            }
        }
        return [$searchCriteria];
    }

    /**
     * Retrieves the numeric quote id from the masked quote id.
     *
     * @param string $maskedId the masked quote id
     * @return string the numeric quote id
     * @throws NoSuchEntityException
     */
    private function getQuoteFromMasked(string $maskedId): string
    {
        $connection = $this->resourceConnection->getConnection();
        $quoteMaskTable = $this->resourceConnection->getTableName('quote_id_mask');

        $query = $connection
            ->select()
            ->from(
                ['qm' => $quoteMaskTable],
                ['qm.quote_id']
            )
            ->where('qm.masked_id = ?', $maskedId);

        $quoteId = $connection->fetchOne($query);

        if (!$quoteId) {
            throw new NoSuchEntityException(__('masked_quote_id not found'));
        }
        return $quoteId;
    }
}
