<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\OutOfProcessPaymentMethods\Model;

use Magento\Framework\Exception\NoSuchEntityException;
use Magento\OutOfProcessPaymentMethods\Api\Data\OutOfProcessPaymentMethodInterface;
use Magento\OutOfProcessPaymentMethods\Api\OutOfProcessPaymentMethodRepositoryInterface;
use Magento\OutOfProcessPaymentMethods\Model\ResourceModel\OutOfProcessPaymentMethod as ResourceModel;
use Magento\OutOfProcessPaymentMethods\Model\ResourceModel\OutOfProcessPaymentMethod\CollectionFactory;

class OutOfProcessPaymentMethodRepository implements OutOfProcessPaymentMethodRepositoryInterface
{
    /**
     * OutOfProcessPaymentMethodRepository constructor.
     *
     * @param ResourceModel                    $resourceModel
     * @param OutOfProcessPaymentMethodFactory $paymentMethodFactory
     * @param CollectionFactory                $collectionFactory
     */
    public function __construct(
        private readonly ResourceModel                    $resourceModel,
        private readonly OutOfProcessPaymentMethodFactory $paymentMethodFactory,
        private readonly CollectionFactory                $collectionFactory,
    ) {
    }

    /**
     * @inheritDoc
     */
    public function getByCode(string $code): OutOfProcessPaymentMethodInterface
    {
        $paymentMethod = $this->findPaymentMethod($code);
        if ($paymentMethod === null) {
            throw new NoSuchEntityException(__('Out of process payment method not found.'));
        }
        return $paymentMethod;
    }

    /**
     * @inheritDoc
     */
    public function getList(): array
    {
        $collection = $this->collectionFactory->create();
        return $collection->getItems();
    }

    /**
     * @inheritDoc
     */
    public function createOrUpdate(OutOfProcessPaymentMethodInterface $payment_method):
        OutOfProcessPaymentMethodInterface
    {
        $existingPaymentMethod = $this->findPaymentMethod($payment_method->getCode());
        if ($existingPaymentMethod) {
            $payment_method->setId($existingPaymentMethod->getId());
        } else {
            $payment_method->setId(null);
        }
        $this->resourceModel->save($payment_method);

        return $this->findPaymentMethod($payment_method->getCode());
    }

    /**
     * @inheritDoc
     */
    private function findPaymentMethod(string $code): ?OutOfProcessPaymentMethod
    {
        $paymentMethod = $this->paymentMethodFactory->create();
        $this->resourceModel->load($paymentMethod, $code, OutOfProcessPaymentMethodInterface::FIELD_CODE);
        return $paymentMethod->isObjectNew() ? null : $paymentMethod;
    }
}
