<?php
/**
 * Copyright 2015 Adobe
 * All Rights Reserved.
 */

namespace Magento\MediaStorage\Helper\File;

use Magento\Framework\App\Filesystem\DirectoryList;

class Media extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    protected $_date;

    /**
     * @var \Magento\Framework\Filesystem
     */
    protected $filesystem;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Framework\Filesystem $filesystem
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\Filesystem $filesystem
    ) {
        parent::__construct($context);
        $this->_date = $date;
        $this->filesystem = $filesystem;
    }

    /**
     * Collect file info
     *
     * Return array(
     *  filename    => string
     *  content     => string|bool
     *  update_time => string
     *  directory   => string
     *
     * @param string $mediaDirectory
     * @param string $path
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function collectFileInfo($mediaDirectory, $path)
    {
        $path = $path !== null ? ltrim($path, '\\/') : '';
        $fullPath = $mediaDirectory . '/' . $path;

        $dir = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA);
        $relativePath = $dir->getRelativePath($fullPath);
        if (!$dir->isFile($relativePath)) {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('The "%1" file doesn\'t exist. Verify the file and try again.', $fullPath)
            );
        }
        if (!$dir->isReadable($relativePath)) {
            throw new \Magento\Framework\Exception\LocalizedException(__('File %1 is not readable', $fullPath));
        }

        $path = str_replace(['/', '\\'], '/', $path);
        // phpcs:disable Magento2.Functions.DiscouragedFunction
        $directory = dirname($path);
        if ($directory == '.') {
            $directory = null;
        }

        return [
            'filename' => basename($path),
            'content' => $dir->readFile($relativePath),
            'update_time' => $this->_date->date(),
            'directory' => $directory
        ];
    }
}
