<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearch\Block;

use Magento\Framework\App\ProductMetadata;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\CurrencyInterface;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use Magento\LiveSearch\Api\ServiceClientInterface;
use Magento\LiveSearch\Model\ModuleVersionReader;
use Magento\ServicesId\Model\ServicesConfigInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Customer\Model\Session;

class BaseSaaSContext extends Template
{
    /**
     * @var ServicesConfigInterface
     */
    private $servicesConfig;

    /**
     * @var ProductMetadata
     */
    private $productMetadata;

    /**
     * @var ModuleVersionReader
     */
    private $moduleVersionReader;

    /**
     * @var ServiceClientInterface
     */
    private $serviceClient;

    /**
     * @var CurrencyInterface
     */
    private $localeCurrency;

    /**
     * @var Session
     */
    private $customerSession;

    /**
     * @param Context $context
     * @param ServicesConfigInterface $servicesConfig
     * @param ProductMetadata $productMetadata
     * @param ModuleVersionReader $moduleVersionReader
     * @param ServiceClientInterface $serviceClient
     * @param CurrencyInterface $localeCurrency
     * @param Session|null $customerSession
     */
    public function __construct(
        Context $context,
        ServicesConfigInterface $servicesConfig,
        ProductMetadata $productMetadata,
        ModuleVersionReader $moduleVersionReader,
        ServiceClientInterface $serviceClient,
        CurrencyInterface $localeCurrency,
        ?Session $customerSession = null
    ) {
        $this->servicesConfig = $servicesConfig;
        $this->productMetadata = $productMetadata;
        $this->moduleVersionReader = $moduleVersionReader;
        $this->serviceClient = $serviceClient;
        $this->localeCurrency = $localeCurrency;
        $this->customerSession = $customerSession ??
            ObjectManager::getInstance()->get(Session::class);
        parent::__construct($context);
    }

    /**
     * Get website code from store view code
     *
     * @return string
     * @throws NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getWebsiteCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $websiteId = $this->_storeManager->getStore($storeId)->getWebsiteId();
        return $this->_storeManager->getWebsite($websiteId)->getCode();
    }

    /**
     * Get store code from store view code.
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getStoreCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $storeGroupId = $this->_storeManager->getStore($storeId)->getStoreGroupId();
        return $this->_storeManager->getGroup($storeGroupId)->getCode();
    }

    /**
     * Get store view code from store switcher
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getStoreViewCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        return $this->_storeManager->getStore($storeId)->getCode();
    }

    /**
     * Get store currency symbol
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getCurrencySymbol(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $currencyCode = $this->_storeManager->getStore($storeId)->getCurrentCurrency()->getCurrencyCode();
        $currency = $this->localeCurrency->getCurrency($currencyCode);
        $currencySymbol = $currency->getSymbol() ?: $currency->getShortName();

        if (empty($currencySymbol)) {
            return $currencyCode;
        }

        return $currencySymbol;
    }

    /**
     * Get Environment Id from Services Id configuration
     *
     * @return string|null
     */
    public function getEnvironmentId(): ?string
    {
        return $this->servicesConfig->getEnvironmentId();
    }

    /**
     * Get Magento edition
     *
     * @return string|null
     */
    public function getMagentoEdition(): ?string
    {
        return $this->productMetadata->getEdition();
    }

    /**
     * Get Magento version
     *
     * @return string|null
     */
    public function getMagentoVersion(): ?string
    {
        return $this->productMetadata->getVersion();
    }

    /**
     * Get module version
     *
     * @return string|null
     */
    public function getModuleVersion(): ?string
    {
        return $this->moduleVersionReader->getVersion();
    }

    /**
     * Get customer group code.
     *
     * @return string
     */
    public function getCustomerGroupCode(): string
    {
        try {
            // Using customerSession to get customer group id
            // Created ticket https://jira.corp.adobe.com/browse/AC-6741 for UserContextInterface
            $customerGroupId = $this->customerSession->getCustomerGroupId();
        } catch (\Magento\Framework\Exception\LocalizedException $e) {
            $customerGroupId = \Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID;
        }

        // sha1 is used in exported data, so we need to pass the same format here
        // https://github.com/magento-commerce/commerce-data-export-ee/blob/302dbb6216373110a8d2a3e636d6d8924f5e0f96/ProductOverrideDataExporter/etc/query.xml#L13
        return \sha1((string)$customerGroupId);
    }
}
