<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearch\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\ServicesId\Model\ServicesClientInterface;
use Magento\ServicesId\Model\ServicesConfigInterface;
use Psr\Log\LoggerInterface;

class MerchantRegistryClient
{
    private const REGISTRY_PATH = 'services_connector/services_id/registry_api_path';
    private const PREMIUM_SEARCH_FEATURE = 'PREMIUM_SEARCH';

    /**
     * @var ServicesConfigInterface
     */
    private $servicesConfig;

    /**
     * @var ServicesClientInterface
     */
    private $servicesClient;

    /**
     * @var ScopeConfigInterface
     */
    private $config;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param ServicesConfigInterface $servicesConfig
     * @param ServicesClientInterface $servicesClient
     * @param ScopeConfigInterface $config
     * @param LoggerInterface $logger
     */
    public function __construct(
        ServicesConfigInterface $servicesConfig,
        ServicesClientInterface $servicesClient,
        ScopeConfigInterface $config,
        LoggerInterface $logger
    ) {
        $this->servicesConfig = $servicesConfig;
        $this->servicesClient = $servicesClient;
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * Call registry API to add Live Search feature
     *
     * @param string $environmentId
     */
    public function register(string $environmentId)
    {
        if ($this->servicesConfig->isApiKeySet() && !empty($environmentId)) {
            try {
                $response = $this->servicesClient->request('PUT', $this->getUrl($environmentId));
                if ($response && !empty($response['status']) && $response['status'] != 200 && !empty($response['message'])) {
                    $this->logger->error('Unable to register for live search.', ['error' => $response['message']]);
                }
            } catch (\Exception $exception) {
                $this->logger->error('Unable to register for live search.', ['error' => $exception]);
            }
        }
    }

    /**
     * @param string $environmentId
     * @return string
     */
    private function getUrl(string $environmentId): string
    {
        $route = $this->config->getValue(self::REGISTRY_PATH);
        $apiVersion = $this->servicesConfig->getRegistryApiVersion();
        $url = sprintf(
            '/%s/%s/%s/%s/%s/%s/%s',
            $route,
            $apiVersion,
            'registry',
            'environments',
            $environmentId,
            'feature',
            self::PREMIUM_SEARCH_FEATURE
        );
        return $url;
    }
}
