<?php
/**
 * Copyright © Magento. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearch\Block\Adminhtml;

use Magento\Framework\App\ProductMetadata;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use Magento\LiveSearch\Api\ServiceClientInterface;
use Magento\LiveSearch\Model\ModuleVersionReader;
use Magento\ServicesConnector\Api\KeyNotFoundException;
use Magento\ServicesId\Model\ServicesConfigInterface;
use Magento\Store\Model\ScopeInterface;

class Index extends Template
{
    /**
     * Config Paths
     * @var string
     */
    private const FRONTEND_URL_PATH = 'live_search/frontend_url';

    /**
     * @var ServicesConfigInterface
     */
    private $servicesConfig;

    /**
     * @var ProductMetadata
     */
    private $productMetadata;

    /**
     * @var ModuleVersionReader
     */
    private $moduleVersionReader;

    /**
     * @var ServiceClientInterface
     */
    private $serviceClient;

    /**
     * Index constructor.
     *
     * @param Context $context
     * @param ServicesConfigInterface $servicesConfig
     * @param ProductMetadata $productMetadata
     * @param ModuleVersionReader $moduleVersionReader
     * @param ServiceClientInterface $serviceClient
     */
    public function __construct(
        Context $context,
        ServicesConfigInterface $servicesConfig,
        ProductMetadata $productMetadata,
        ModuleVersionReader $moduleVersionReader,
        ServiceClientInterface $serviceClient
    ) {
        $this->servicesConfig = $servicesConfig;
        $this->productMetadata = $productMetadata;
        $this->moduleVersionReader = $moduleVersionReader;
        $this->serviceClient = $serviceClient;
        parent::__construct($context);
    }

    /**
     * Returns config for frontend url
     *
     * @return string
     */
    public function getFrontendUrl(): string
    {
        return (string) $this->_scopeConfig->getValue(
            self::FRONTEND_URL_PATH,
            ScopeInterface::SCOPE_WEBSITE
        );
    }

    /**
     * Get website code from store view code
     *
     * @return string
     * @throws NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getWebsiteCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $websiteId = $this->_storeManager->getStore($storeId)->getWebsiteId();
        return $this->_storeManager->getWebsite($websiteId)->getCode();
    }

    /**
     * Get store code from store view code.
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getStoreCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $storeGroupId = $this->_storeManager->getStore($storeId)->getStoreGroupId();
        return $this->_storeManager->getGroup($storeGroupId)->getCode();
    }

    /**
     * Get store view code from store switcher
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getStoreViewCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        return $this->_storeManager->getStore($storeId)->getCode();
    }

    /**
     * Get Environment Id from Services Id configuration
     *
     * @return string|null
     */
    public function getEnvironmentId(): ?string
    {
        return $this->servicesConfig->getEnvironmentId();
    }

    /**
     * Get Environment Name from Services Id configuration
     *
     * @return string|null
     */
    public function getEnvironmentName(): ?string
    {
        return $this->servicesConfig->getEnvironmentName();
    }

    /**
     * Get Environment Type from Services Id configuration
     *
     * @return string|null
     */
    public function getEnvironmentType(): ?string
    {
        return $this->servicesConfig->getEnvironment();
    }

    /**
     * Get Magento edition
     *
     * @return string|null
     */
    public function getMagentoEdition(): ?string
    {
        return $this->productMetadata->getEdition();
    }

    /**
     * Get Magento version
     *
     * @return string|null
     */
    public function getMagentoVersion(): ?string
    {
        return $this->productMetadata->getVersion();
    }

    /**
     * Get module version
     *
     * @return string|null
     */
    public function getModuleVersion(): ?string
    {
        return $this->moduleVersionReader->getVersion();
    }

    /**
     * Check if API key is valid
     *
     * @return bool
     */
    public function isApiKeyValid(): bool
    {
        try {
            return $this->serviceClient->isApiKeyValid();
        } catch (KeyNotFoundException $exception) {
            return false;
        }
    }
}
