<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor\FilterHandler;

use Exception;
use GraphQL\Language\AST\NodeKind;
use GraphQL\Language\Parser;
use GraphQL\Language\Visitor;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Framework\Api\Search\SearchCriteriaInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\LiveSearchAdapter\Model\CategoryCache;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class CategoryFilterHandler implements FilterHandlerInterface
{
    /**
     * @var array
     */
    private array $filterValues;

    /**
     * @var CategoryRepositoryInterface
     */
    private CategoryRepositoryInterface $categoryRepository;

    /**
     * @var StoreManagerInterface
     */
    private StoreManagerInterface $storeManager;

    /**
     * @var CategoryCache
     */
    private CategoryCache $categoryCache;

    /**
     * @var SearchCriteriaInterface
     */
    private SearchCriteriaInterface $searchCriteria;

    /**
     * @var RequestInterface
     */
    private RequestInterface $request;

    /**
     * @var SerializerInterface
     */
    private SerializerInterface $jsonSerializer;

    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @param array $filterValues
     * @param SearchCriteriaInterface $searchCriteria
     * @param CategoryRepositoryInterface $categoryRepository
     * @param StoreManagerInterface $storeManager
     * @param CategoryCache $categoryCache
     * @param RequestInterface $request
     * @param SerializerInterface $jsonSerializer
     * @param LoggerInterface $logger
     */
    public function __construct(
        array $filterValues,
        SearchCriteriaInterface $searchCriteria,
        CategoryRepositoryInterface $categoryRepository,
        StoreManagerInterface $storeManager,
        CategoryCache $categoryCache,
        RequestInterface $request,
        SerializerInterface $jsonSerializer,
        LoggerInterface $logger
    ) {
        $this->filterValues = $filterValues;
        $this->searchCriteria = $searchCriteria;
        $this->categoryRepository = $categoryRepository;
        $this->storeManager = $storeManager;
        $this->categoryCache = $categoryCache;
        $this->request = $request;
        $this->jsonSerializer = $jsonSerializer;
        $this->logger = $logger;
    }

    /**
     * @inheritdoc
     */
    public function getFilterKey(): string
    {
        return 'categories';
    }

    /**
     * Get filter variables
     *
     * @return array|array[]
     * @throws NoSuchEntityException
     */
    public function getFilterVariables(): array
    {
        $categoryIds = array_unique($this->filterValues);
        if (empty($categoryIds)) {
            return [];
        }
        $categories = $this->getCategoryData($categoryIds);

        if ($this->isRequestForCategoryPage()) {
            $categoriesAtTopMostLevel = $this->getCategoriesAtTopMostLevel($categories);
            $categoriesAtTopMostLevel = array_values($categoriesAtTopMostLevel);

            if (count($categoriesAtTopMostLevel) === 1) {
                $topMostCategory = $categoriesAtTopMostLevel[0];
                $subCategories = $this->getSubCategories($topMostCategory, $categories);

                // if all the remaining categories in filter are subcategories of the top most category
                // then it is a valid browse request
                $isValidBrowseRequest = count($subCategories) === count($categories) - 1;

                if ($isValidBrowseRequest) {
                    $childCategoryPaths = array_column($subCategories, 'path');
                    $filterVariables[] = ['attribute' => 'categoryPath', 'eq' => $topMostCategory['path']];
                    if (count($childCategoryPaths) > 0) {
                        $filterVariables[] = [
                            'attribute' => $this->getFilterKey(), $this->getFilterType() => $childCategoryPaths
                        ];
                    }
                    return $filterVariables;
                }
            }
        }

        return [
            [
                'attribute' => $this->getFilterKey(),
                $this->getFilterType() => array_column($categories, 'path')
            ]
        ];
    }

    /**
     * Get categories at top most level
     *
     * @param array $categories
     * @return array
     */
    private function getCategoriesAtTopMostLevel(array $categories): array
    {
        $levels = array_column($categories, 'level');
        $topMostLevel = min($levels);
        return array_filter($categories, function ($category) use ($topMostLevel) {
            return $category['level'] === $topMostLevel;
        });
    }

    /**
     * Get sub categories of $topMostCategory
     *
     * @param array $topMostCategory
     * @param array $categories
     * @return array
     */
    private function getSubCategories(array $topMostCategory, array $categories): array
    {
        $subCategories = [];
        foreach ($categories as $category) {
            if ($category['entity_id'] !== $topMostCategory['entity_id'] &&
                str_contains($category['id_path'], $topMostCategory['id_path'])) {
                $subCategories[] = $category;
            }
        }
        return $subCategories;
    }

    /**
     * Is request for category page
     *
     * @return bool
     */
    private function isRequestForCategoryPage(): bool
    {
        // check for category page - REST
        if ($this->searchCriteria->getRequestName() === 'catalog_view_container') {
            return true;
        }

        // check for category page - graphql
        if (\str_contains($this->searchCriteria->getRequestName(), 'graphql')) {
            return $this->isRequestForCategoryPageGraphql();
        }

        return false;
    }

    /**
     * Is request for category page using graphql query
     *
     * @return bool
     */
    private function isRequestForCategoryPageGraphql(): bool
    {
        $operationName = \strtolower($this->request->getParam('operationName', ''));
        if (!empty($operationName) &&
            (\str_contains($operationName, 'categories') || \str_contains($operationName, 'categoryList'))) {
            return true;
        }

        $operations = [];
        try {
            $graphQlRequest = $this->jsonSerializer->unserialize($this->request->getContent());
            Visitor::visit(Parser::parse($graphQlRequest['query']), [
                NodeKind::OPERATION_DEFINITION => function ($node) use (&$operations) {
                    $selections = \array_map(function ($selection) {
                        return $selection['name']['value'];
                    }, $node->toArray(true)['selectionSet']['selections']);

                    foreach ($selections as $selection) {
                        $operations[] = $selection;
                    }

                    return Visitor::stop();
                }
            ]);
        } catch (Exception $e) {
            $this->logger->warning(
                'LiveSearchAdapter: Unable to parse graphql operation name. Error: ' . $e->getMessage()
            );
        }

        return !empty(\array_intersect(['categories', 'categoryList'], $operations));
    }

    /**
     * Get category ids
     *
     * @param array $ids
     * @return array
     * @throws NoSuchEntityException
     */
    private function getCategoryData(array $ids): array
    {
        $storeId = $this->storeManager->getStore()->getId();
        $categoryIdsNotInCache = [];
        $categories = [];

        foreach ($ids as $id) {
            $key = $id . '_' . $storeId;
            $categoryData = $this->categoryCache->load($key);
            if ($categoryData) {
                $categories[] = $categoryData;
            } else {
                $categoryIdsNotInCache[] = $id;
            }
        }

        if (count($categoryIdsNotInCache) > 0) {
            foreach ($categoryIdsNotInCache as $categoryId) {
                $category = $this->categoryRepository->get($categoryId, $storeId);
                $category->getUrlInstance()->setScope($storeId);
                $categoryData = [
                    'entity_id' => $categoryId,
                    'parent_id' => $category->getParentId(),
                    'id_path' => $category->getPath(),
                    'path' => $category->getUrlPath(),
                    'level' => (int) $category->getLevel(),
                ];
                $key = $categoryData['entity_id'] . '_' . $storeId;
                $this->categoryCache->save($key, $categoryData);
                $categories[] = $categoryData;
            }
        }

        return $categories;
    }

    /**
     * @inheritdoc
     */
    public function getFilterType(): string
    {
        return 'in';
    }
}
