<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor\FilterHandler;

use Magento\Framework\Api\Search\SearchCriteriaInterface;

class FilterHandlerFactory
{
    /**
     * @var RangeFilterHandlerFactory
     */
    private $rangeFilterHandlerFactory;

    /**
     * @var CategoryFilterHandlerFactory
     */
    private $categoryFilterHandlerFactory;

    /**
     * @var VisibilityFilterHandlerFactory
     */
    private $visibilityFilterHandlerFactory;

    /**
     * @var AttributeFilterHandlerFactory
     */
    private $attributeFilterHandlerFactory;

    /**
     * @var DefaultAttributeFilterHandlerFactory
     */
    private $defaultAttributeFilterHandlerFactory;

    /**
     * FilterHandlerFactory constructor.
     * @param RangeFilterHandlerFactory $rangeFilterHandlerFactory
     * @param CategoryFilterHandlerFactory $categoryFilterHandlerFactory
     * @param VisibilityFilterHandlerFactory $visibilityFilterHandlerFactory
     * @param AttributeFilterHandlerFactory $attributeFilterHandlerFactory
     * @param DefaultAttributeFilterHandlerFactory $defaultAttributeFilterHandlerFactory
     */
    public function __construct(
        RangeFilterHandlerFactory $rangeFilterHandlerFactory,
        CategoryFilterHandlerFactory $categoryFilterHandlerFactory,
        VisibilityFilterHandlerFactory $visibilityFilterHandlerFactory,
        AttributeFilterHandlerFactory $attributeFilterHandlerFactory,
        DefaultAttributeFilterHandlerFactory $defaultAttributeFilterHandlerFactory
    ) {
        $this->rangeFilterHandlerFactory = $rangeFilterHandlerFactory;
        $this->categoryFilterHandlerFactory = $categoryFilterHandlerFactory;
        $this->visibilityFilterHandlerFactory = $visibilityFilterHandlerFactory;
        $this->attributeFilterHandlerFactory = $attributeFilterHandlerFactory;
        $this->defaultAttributeFilterHandlerFactory = $defaultAttributeFilterHandlerFactory;
    }

    /**
     * Resolve handler
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @param string $filterKey
     * @param mixed $filterValue
     * @return FilterHandlerInterface
     */
    public function resolve(
        SearchCriteriaInterface $searchCriteria,
        string $filterKey,
        $filterValue
    ): FilterHandlerInterface {
        if (is_array($filterValue) && !empty(array_intersect(array_keys($filterValue), ['from', 'to']))) {
            return $this->rangeFilterHandlerFactory->create(
                ['filterKey' => $filterKey, 'filterValues' => $filterValue]
            );
        }

        switch ($filterKey) {
            case 'category_ids':
            case 'category_id':
                $handler = $this->categoryFilterHandlerFactory->create(
                    ['filterKey' => $filterKey, 'filterValues' => $filterValue, 'searchCriteria' => $searchCriteria]
                );
                break;
            case 'visibility':
                $handler = $this->visibilityFilterHandlerFactory->create(
                    ['filterKey' => $filterKey, 'filterValues' => $filterValue]
                );
                break;
            case 'url_key':
            case 'sku':
                $handler = $this->defaultAttributeFilterHandlerFactory->create(
                    ['filterKey' => $filterKey, 'filterValues' => $filterValue]
                );
                break;
            default:
                $handler = $this->attributeFilterHandlerFactory->create(
                    ['filterKey' => $filterKey, 'filterValues' => $filterValue]
                );
        }

        return $handler;
    }
}
