<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor;

use GraphQL\QueryBuilder\QueryBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\App\ObjectManager;
use Magento\LiveSearch\Block\BaseSaaSContext;

/**
 * QueryArgumentProcessor for Context argument
 */
class ContextQueryArgumentProcessor implements QueryArgumentProcessorInterface
{
    /**
     * @var string
     */
    private const ARGUMENT_NAME = 'context';

    /**
     * @var string
     */
    private const TYPE = 'QueryContextInput!';

    /**
     * @var bool
     */
    private const IS_REQUIRED = false;

    /**
     * @var string
     */
    private const ARGUMENT_VALUE = '$context';

    /**
     * @var UserContextInterface
     */
    private $userContext;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var BaseSaaSContext
     */
    private $baseSaaSContext;

    /**
     * @param UserContextInterface $userContext
     * @param CustomerRepositoryInterface $customerRepository
     * @param BaseSaaSContext|null $baseSaaSContext
     */
    public function __construct(
        UserContextInterface $userContext,
        CustomerRepositoryInterface $customerRepository,
        ?BaseSaaSContext $baseSaaSContext = null
    ) {
        $this->userContext = $userContext;
        $this->customerRepository = $customerRepository;
        $this->baseSaaSContext = $baseSaaSContext ??
            ObjectManager::getInstance()->get(BaseSaaSContext::class);
    }

    /**
     * @inheritdoc
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @return mixed
     */
    public function getQueryArgumentValue(SearchCriteriaInterface $searchCriteria)
    {
        return [
            'customerGroup' => $this->getCustomerGroupCode()
        ];
    }

    /**
     * Get customer group code.
     *
     * @return string
     */
    private function getCustomerGroupCode(): string
    {
        return $this->baseSaaSContext->getCustomerGroupCode();
    }

    /**
     * @inheritdoc
     *
     * @param QueryBuilder $builder
     */
    public function setQueryArguments(QueryBuilder $builder): void
    {
        $builder->setVariable(self::ARGUMENT_NAME, self::TYPE, self::IS_REQUIRED);
        $builder->setArgument(self::ARGUMENT_NAME, self::ARGUMENT_VALUE);
    }

    /**
     * @inheritdoc
     *
     * @return bool
     */
    public function isRequired(): bool
    {
        return self::IS_REQUIRED;
    }
}
