<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\Aggregation;


class BucketHandlerFactory
{
    /**
     * @var CategoryBucketHandlerFactory
     */
    private $categoryBucketHandlerFactory;

    /**
     * @var PriceBucketHandlerFactory
     */
    private $priceBucketHandlerFactory;

    /**
     * @var AttributeBucketHandlerFactory
     */
    private $attributeBucketHandlerFactory;

    /**
     * BucketHandlerFactory constructor.
     * @param CategoryBucketHandlerFactory $categoryBucketHandlerFactory
     * @param PriceBucketHandlerFactory $priceBucketHandlerFactory
     * @param AttributeBucketHandlerFactory $attributeBucketHandlerFactory
     */
    public function __construct(
        CategoryBucketHandlerFactory $categoryBucketHandlerFactory,
        PriceBucketHandlerFactory $priceBucketHandlerFactory,
        AttributeBucketHandlerFactory $attributeBucketHandlerFactory
    ) {
        $this->categoryBucketHandlerFactory = $categoryBucketHandlerFactory;
        $this->priceBucketHandlerFactory = $priceBucketHandlerFactory;
        $this->attributeBucketHandlerFactory = $attributeBucketHandlerFactory;
    }

    /**
     * @param string $bucketName
     * @param array $rawBuckets
     * @param array $attributesMetadata
     * @param string $storeViewCode
     * @return BucketHandlerInterface
     */
    public function resolve(string $bucketName, array $rawBuckets, array $attributesMetadata, string $storeViewCode): BucketHandlerInterface
    {
        switch ($bucketName) {
            case 'categories':
                $handler = $this->categoryBucketHandlerFactory->create(['rawBuckets' => $rawBuckets]);
                break;
            case 'price':
                $handler = $this->priceBucketHandlerFactory->create(['rawBuckets' => $rawBuckets]);
                break;
            default:
                $handler = $this->attributeBucketHandlerFactory->create([
                    'storeViewCode' => $storeViewCode,
                    'attributeCode' => $bucketName,
                    'rawBuckets' => $rawBuckets,
                    'attributesMetadata' => $attributesMetadata
                ]);
        }

        return $handler;
    }
}
