<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor;

use GraphQL\QueryBuilder\QueryBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;

/**
 * QueryArgumentProcessor for Context argument
 */
class ContextQueryArgumentProcessor implements QueryArgumentProcessorInterface
{
    /**
     * @var string
     */
    private const ARGUMENT_NAME = 'context';

    /**
     * @var string
     */
    private const TYPE = 'QueryContextInput!';

    /**
     * @var bool
     */
    private const IS_REQUIRED = false;

    /**
     * @var string
     */
    private const ARGUMENT_VALUE = '$context';

    /**
     * @var UserContextInterface
     */
    private $userContext;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @param UserContextInterface $userContext
     * @param CustomerRepositoryInterface $customerRepository
     */
    public function __construct(
        UserContextInterface $userContext,
        CustomerRepositoryInterface $customerRepository
    ) {
        $this->userContext = $userContext;
        $this->customerRepository = $customerRepository;
    }

    /**
     * @inheritdoc
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @return mixed
     */
    public function getQueryArgumentValue(SearchCriteriaInterface $searchCriteria)
    {
        return [
            'customerGroup' => $this->getCustomerGroupCode()
        ];
    }

    /**
     * Get customer group code.
     *
     * @return string
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function getCustomerGroupCode(): string
    {
        try {
            $customer = $this->customerRepository->getById($this->userContext->getUserId());
            $customerGroupId = $customer->getGroupId();
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $customerGroupId = \Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID;
        }

        // sha1 is used in exported data, so we need to pass the same format here
        // https://github.com/magento-commerce/commerce-data-export-ee/blob/302dbb6216373110a8d2a3e636d6d8924f5e0f96/ProductOverrideDataExporter/etc/query.xml#L13
        return \sha1((string)$customerGroupId);
    }

    /**
     * @inheritdoc
     *
     * @param QueryBuilder $builder
     */
    public function setQueryArguments(QueryBuilder $builder): void
    {
        $builder->setVariable(self::ARGUMENT_NAME, self::TYPE, self::IS_REQUIRED);
        $builder->setArgument(self::ARGUMENT_NAME, self::ARGUMENT_VALUE);
    }

    /**
     * @inheritdoc
     *
     * @return bool
     */
    public function isRequired(): bool
    {
        return self::IS_REQUIRED;
    }
}
