<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model;

use Magento\Catalog\Model\Category;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Serialize\SerializerInterface;

class CategoryCache
{
    /**
     * Cache key for store label attribute
     */
    private const CACHE_PREFIX = 'LIVE_SEARCH_CATEGORY_DATA';

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @param CacheInterface $cache
     * @param SerializerInterface $serializer
     * @codeCoverageIgnore
     */
    public function __construct(
        CacheInterface $cache,
        SerializerInterface $serializer
    ) {
        $this->cache = $cache;
        $this->serializer = $serializer;
    }

    /**
     * @param int $categoryId
     * @return array|null
     */
    public function load(int $categoryId): ?array
    {
        $cacheKey = self::CACHE_PREFIX . $categoryId;
        if ($categoryData = $this->cache->load($cacheKey)) {
            return $this->serializer->unserialize($categoryData);
        }
        return null;
    }

    /**
     * @param int $categoryId
     * @param array $categoryData
     */
    public function save(int $categoryId, array $categoryData): void
    {
        $cacheKey = self::CACHE_PREFIX . $categoryId;
        $this->cache->save(
            $this->serializer->serialize($categoryData),
            $cacheKey,
            [
                Category::CACHE_TAG
            ]
        );
    }
}
