<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model;

use Magento\Framework\Api\Search\SearchCriteriaInterface;
use Magento\Framework\Api\Search\SearchResultInterface;
use Magento\Search\Api\SearchInterface;
use Psr\Log\LoggerInterface;

/**
 * Class SearchAdapter
 *
 * Overloads control under search requests
 */
class SearchAdapter implements SearchInterface
{
    /**
     * @var QueryBuilder
     */
    private $queryBuilder;

    /**
     * @var SearchClient
     */
    private $searchClient;

    /**
     * @var SearchResultBuilder
     */
    private $searchResultBuilder;

    /**
     * @var EmptySearchResultBuilder
     */
    private $emptySearchResultBuilder;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * SearchAdapter constructor.
     * @param QueryBuilder $queryBuilder
     * @param SearchClient $searchClient
     * @param SearchResultBuilder $searchResultBuilder
     * @param EmptySearchResultBuilder $emptySearchResultBuilder
     * @param LoggerInterface $logger
     */
    public function __construct(
        QueryBuilder $queryBuilder,
        SearchClient $searchClient,
        SearchResultBuilder $searchResultBuilder,
        EmptySearchResultBuilder $emptySearchResultBuilder,
        LoggerInterface $logger
    ) {
        $this->queryBuilder = $queryBuilder;
        $this->searchClient = $searchClient;
        $this->searchResultBuilder = $searchResultBuilder;
        $this->emptySearchResultBuilder = $emptySearchResultBuilder;
        $this->logger = $logger;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return SearchResultInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function search(SearchCriteriaInterface $searchCriteria): SearchResultInterface
    {
        try {
            if (empty($searchCriteria->getSortOrders())) {
                $searchCriteria->setSortOrders([]);
            }

            $body = $this->queryBuilder->build($searchCriteria);
            $saasResult = $this->searchClient->request($body);
            $searchResult = $this->searchResultBuilder->build($saasResult);
        } catch (\Exception $exception) {
            $this->logger->error('LiveSearchAdapter: An error occurred: ' . $exception->getMessage());
            $searchResult = $this->emptySearchResultBuilder->build();
        }

        $searchResult->setSearchCriteria($searchCriteria);

        return $searchResult;
    }
}
