<?php

/**
 * Copyright © Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor;

use GraphQL\QueryBuilder\QueryBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;

class SortQueryArgumentProcessor implements QueryArgumentProcessorInterface
{
    /**
     * @var string
     */
    private const ARGUMENT_NAME = 'sort';

    /**
     * @var string
     */
    private const TYPE = '[ProductSearchSortInput!]';

    /**
     * @var bool
     */
    private const IS_REQUIRED = false;

    /**
     * @var string
     */
    private const ARGUMENT_VALUE = '$sort';

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return array|mixed
     */
    public function getQueryArgumentValue(SearchCriteriaInterface $searchCriteria)
    {
        if (empty($searchCriteria->getSortOrders())) {
            return [];
        }
        $sortVariables = [];
        foreach ($searchCriteria->getSortOrders() as $sortKey => $sortOrder) {
            /**
             * Sort order array has an attribute like name / price / relevance and entity_id.
             * entity_id is used in addition to the primary sort attribute in case of a tie.
             * We are ignoring sorting by entity_id here.
             */
            if (!empty($sortKey) && !in_array($sortKey, ['entity_id'], true)) {
                $sortVariables = [
                    'attribute' => $sortKey,
                    'direction' => $sortOrder
                ];
            }
        }

        return $sortVariables;
    }

    /**
     * @param QueryBuilder $builder
     */
    public function setQueryArguments(QueryBuilder $builder): void
    {
        $builder->setVariable(self::ARGUMENT_NAME, self::TYPE, self::IS_REQUIRED);
        $builder->setArgument(self::ARGUMENT_NAME, self::ARGUMENT_VALUE);
    }

    /**
     * @return bool
     */
    public function isRequired(): bool
    {
        return self::IS_REQUIRED;
    }
}
