<?php

/**
 * Copyright © Magento. All rights reserved.
 * See COPYING.txt for license details.
 */

declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor;

use GraphQL\QueryBuilder\QueryBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\LiveSearchAdapter\Model\AttributeMetadata;
use Magento\LiveSearchAdapter\Model\QueryArgumentProcessor\FilterHandler\FilterHandlerFactory;

class FilterQueryArgumentProcessor implements QueryArgumentProcessorInterface
{
    /**
     * @var string
     */
    private const ARGUMENT_NAME = 'filter';

    /**
     * @var string
     */
    private const TYPE = '[SearchClauseInput!]';

    /**
     * @var bool
     */
    private const IS_REQUIRED = false;

    /**
     * @var string
     */
    private const ARGUMENT_VALUE = '$filter';

    /**
     * @var AttributeMetadata
     */
    private $attributeMetadata;

    /**
     * @var FilterHandlerFactory
     */
    private $filterHandlerFactory;

    /**
     * FilterQueryArgumentProcessor constructor.
     * @param AttributeMetadata $attributeMetadata
     * @param FilterHandlerFactory $filterHandlerFactory
     */
    public function __construct(
        AttributeMetadata $attributeMetadata,
        FilterHandlerFactory $filterHandlerFactory
    ) {
        $this->attributeMetadata = $attributeMetadata;
        $this->filterHandlerFactory = $filterHandlerFactory;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return array|mixed
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getQueryArgumentValue(SearchCriteriaInterface $searchCriteria)
    {
        $filters = $this->parseFilters($searchCriteria);

        $filterVariables = [];
        foreach ($filters as $filterKey => $rawFilterValue) {
            $filterHandler = $this->filterHandlerFactory->resolve($filterKey, $rawFilterValue);
            $filterValue = $filterHandler->getFilterValue();
            if (!empty($filterValue)) {
                $filterVariables[] = [
                    'attribute' => $filterHandler->getFilterKey(),
                    $filterHandler->getFilterType() => $filterValue
                ];
            }
        }

        return $filterVariables;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return array
     */
    private function parseFilters(SearchCriteriaInterface $searchCriteria): array
    {
        $filters = [];
        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                $field = $filter->getField();
                $condition = $filter->getValue();

                if ($this->stringEndsWith($field, '.from')) {
                    $attribute = substr($field, 0, strpos($field, '.from'));
                    $filters[$attribute]['from'] = $condition;
                } elseif ($this->stringEndsWith($field, '.to')) {
                    $attribute = substr($field, 0, strpos($field, '.to'));
                    $filters[$attribute]['to'] = $condition;
                } elseif ($field !== 'price_dynamic_algorithm' && $field !== 'search_term') {
                    $fieldValue = is_array($condition) ? $condition : [$condition];
                    $filters[$field] = isset($filters[$field]) ? array_merge($filters[$field], $fieldValue) : $fieldValue;
                }
            }
        }
        return $filters;
    }

    /**
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    private function stringEndsWith(string $haystack, string $needle): bool
    {
        return substr($haystack, -strlen($needle)) === $needle;
    }

    /**
     * @param QueryBuilder $builder
     */
    public function setQueryArguments(QueryBuilder $builder): void
    {
        $builder->setVariable(self::ARGUMENT_NAME, self::TYPE, self::IS_REQUIRED);
        $builder->setArgument(self::ARGUMENT_NAME, self::ARGUMENT_VALUE);
    }

    /**
     * @return bool
     */
    public function isRequired(): bool
    {
        return self::IS_REQUIRED;
    }
}
