<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model;

use Magento\Framework\Api\AttributeValueFactory;
use Magento\Framework\Api\Search\AggregationInterface;
use Magento\Framework\Api\Search\DocumentFactory;
use Magento\Framework\Api\Search\SearchResultFactory;
use Magento\Framework\Api\Search\SearchResultInterface;
use Magento\Framework\Search\Response\AggregationFactory;
use Magento\LiveSearchAdapter\Model\Aggregation\AttributeEmptyBucketHandlerFactory;
use Magento\LiveSearchAdapter\Model\Aggregation\BucketHandlerFactory;

class SearchResultBuilder
{
    /**
     * @var SearchResultFactory
     */
    private $searchResultFactory;

    /**
     * @var DocumentFactory
     */
    private $documentFactory;

    /**
     * @var BucketHandlerFactory
     */
    private $bucketHandlerFactory;

    /**
     * @var AggregationFactory
     */
    private $aggregationFactory;

    /**
     * @var SearchScopeResolver
     */
    private $scopeResolver;

    /**
     * @var AttributeMetadata
     */
    private $attributeMetadata;

    /**
     * @var AttributeValueFactory
     */
    private $attributeValueFactory;

    /**
     * @var FilterableAttributes
     */
    private $filterableAttributes;

    /**
     * @var AttributeEmptyBucketHandlerFactory
     */
    private $attributeEmptyBucketHandlerFactory;

    /**
     * @var EmptySearchResultBuilder
     */
    private $emptySearchResultBuilder;

    /**
     * SearchResultBuilder constructor.
     * @param SearchResultFactory $searchResultFactory
     * @param DocumentFactory $documentFactory
     * @param BucketHandlerFactory $bucketHandlerFactory
     * @param AggregationFactory $aggregationFactory
     * @param SearchScopeResolver $scopeResolver
     * @param AttributeMetadata $attributeMetadata
     * @param AttributeValueFactory $attributeValueFactory
     * @param FilterableAttributes $filterableAttributes
     * @param AttributeEmptyBucketHandlerFactory $attributeEmptyBucketHandlerFactory
     * @param EmptySearchResultBuilder $emptySearchResultBuilder
     */
    public function __construct(
        SearchResultFactory $searchResultFactory,
        DocumentFactory $documentFactory,
        BucketHandlerFactory $bucketHandlerFactory,
        AggregationFactory $aggregationFactory,
        SearchScopeResolver $scopeResolver,
        AttributeMetadata $attributeMetadata,
        AttributeValueFactory $attributeValueFactory,
        FilterableAttributes $filterableAttributes,
        AttributeEmptyBucketHandlerFactory $attributeEmptyBucketHandlerFactory,
        EmptySearchResultBuilder $emptySearchResultBuilder
    ) {
        $this->searchResultFactory = $searchResultFactory;
        $this->documentFactory = $documentFactory;
        $this->bucketHandlerFactory = $bucketHandlerFactory;
        $this->aggregationFactory = $aggregationFactory;
        $this->scopeResolver = $scopeResolver;
        $this->attributeMetadata = $attributeMetadata;
        $this->attributeValueFactory = $attributeValueFactory;
        $this->filterableAttributes = $filterableAttributes;
        $this->attributeEmptyBucketHandlerFactory = $attributeEmptyBucketHandlerFactory;
        $this->emptySearchResultBuilder = $emptySearchResultBuilder;
    }

    /**
     * @param array $saasResult
     * @return SearchResultInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function build(array $saasResult): SearchResultInterface
    {
        if (!isset($saasResult['data'])) {
            return $this->emptySearchResultBuilder->build();
        }
        $searchResult = $this->searchResultFactory->create();
        $searchData = $saasResult['data'];
        $items = $this->getItems($searchData['productSearch']);
        $searchResult->setItems($items);
        $aggregations = $this->getAggregations($searchData['productSearch']);
        $searchResult->setAggregations($aggregations);
        $searchResult->setTotalCount($searchData['productSearch']['total_count']);

        return $searchResult;
    }

    /**
     * @param $searchData
     * @return array
     */
    private function getItems($searchData): array
    {
        $items = [];
        if (isset($searchData['items']) and is_array($searchData['items'])) {
            $score = count($searchData['items']);
            foreach ($searchData['items'] as $item) {
                $attributeScore = $this->attributeValueFactory->create();
                $attributeScore->setAttributeCode('score');
                $attributeScore->setValue($score);
                $document = $this->documentFactory->create();
                $document->setId((int)$item['product']['uid']);
                $document->setCustomAttributes(['score' => $attributeScore]);
                $items[] = $document;
                $score -= 1;
            }
        }
        return $items;
    }

    /**
     * @param $searchData
     * @return AggregationInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getAggregations($searchData): AggregationInterface
    {
        $buckets = [];
        $searchApiAttributeCodes = [];
        if (isset($searchData['facets'])) {
            $attributeCodes =  array_column($searchData['facets'], 'attribute');
            // category and price are special cases and not standard attributes
            $attributeCodes = array_diff($attributeCodes, ['categories', 'price']);
            $storeViewCode = $this->scopeResolver->getStoreViewCode();
            $attributesMetadata = $this->attributeMetadata->getAttributesMetadata($attributeCodes, $storeViewCode);

            foreach ($searchData['facets'] as $facet) {
                $bucketHandler = $this->bucketHandlerFactory->resolve($facet['attribute'], $facet['buckets'], $attributesMetadata, $storeViewCode);
                $bucket = $bucketHandler->getBucket();
                if (!empty($bucket)) {
                    $buckets[$bucketHandler->getBucketName()] = $bucket;
                    $searchApiAttributeCodes[] = $facet['attribute'];
                }
            }
        }

        // create empty buckets for filterable attributes in Magento that are not present in SaaS search results as a
        // workaround to avoid exception in Magento\CatalogSearch\Model\ResourceModel\Fulltext\Collection::getFacetedData()
        $emptyBuckets = $this->createEmptyBucketsForMandatoryFilterableAttributes($searchApiAttributeCodes);

        return $this->aggregationFactory->create([
            'buckets' => array_merge($emptyBuckets, $buckets)
        ]);
    }

    /**
     * @param array $searchApiAttributeCodes
     * @return array
     */
    private function createEmptyBucketsForMandatoryFilterableAttributes(array $searchApiAttributeCodes): array
    {
        $buckets = [];
        $filterableAttributeCodes = $this->filterableAttributes->getMandatoryFilterableAttributesForLayeredNavigation();
        $filterableAttributeCodes = array_diff($filterableAttributeCodes, $searchApiAttributeCodes);

        foreach ($filterableAttributeCodes as $attributeCode) {
            $attributeEmptyBucketHandler = $this->attributeEmptyBucketHandlerFactory->create(['attributeCode' => $attributeCode]);
            $buckets[$attributeEmptyBucketHandler->getBucketName()] = $attributeEmptyBucketHandler->getBucket();
        }

        return $buckets;
    }
}
