<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\LiveSearch\Api\ServiceClient;

/**
 * Class SearchClient
 *
 * Makes request to Search SaaS
 */
class SearchClient
{
    /**
     * Config path
     */
    const BACKEND_PATH = 'live_search_adapter/backend_path';

    /**
     * @var ServiceClient
     */
    private $serviceClient;

    /**
     * @var SearchScopeResolver
     */
    private $scopeResolver;

    /**
     * @var ScopeConfigInterface
     */
    private $config;

    public function __construct(
        ServiceClient $serviceClient,
        SearchScopeResolver $scopeResolver,
        ScopeConfigInterface $config
    ) {
        $this->serviceClient = $serviceClient;
        $this->scopeResolver = $scopeResolver;
        $this->config = $config;
    }

    /**
     * @param string $body
     * @return array
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\LiveSearch\Api\ApiException
     * @throws \Magento\LiveSearch\Api\KeyInvalidException
     */
    public function request(string $body): array
    {
        $headers = $this->getHeaders();
        $path = $this->config->getValue(self::BACKEND_PATH);

        $saasResult = $this->serviceClient->request($headers, $path, $body);

        return $saasResult;
    }

    /**
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getHeaders(): array
    {
        return [
            'Magento-Website-Code' => $this->scopeResolver->getWebsiteCode(),
            'Magento-Store-Code' => $this->scopeResolver->getStoreCode(),
            'Magento-Store-View-Code' => $this->scopeResolver->getStoreViewCode()
        ];
    }
}
