<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model;

use Magento\Framework\Api\Search\AggregationInterface;
use Magento\Framework\Api\Search\SearchResultFactory;
use Magento\Framework\Api\Search\SearchResultInterface;
use Magento\Framework\Search\Response\AggregationFactory;
use Magento\LiveSearchAdapter\Model\Aggregation\AttributeEmptyBucketHandlerFactory;

class EmptySearchResultBuilder
{
    /**
     * @var SearchResultFactory
     */
    private $searchResultFactory;

    /**
     * @var AggregationFactory
     */
    private $aggregationFactory;

    /**
     * @var FilterableAttributes
     */
    private $filterableAttributes;

    /**
     * @var AttributeEmptyBucketHandlerFactory
     */
    private $attributeEmptyBucketHandlerFactory;

    /**
     * SearchResultBuilder constructor.
     * @param SearchResultFactory $searchResultFactory
     * @param AggregationFactory $aggregationFactory
     * @param FilterableAttributes $filterableAttributes
     * @param AttributeEmptyBucketHandlerFactory $attributeEmptyBucketHandlerFactory
     */
    public function __construct(
        SearchResultFactory $searchResultFactory,
        AggregationFactory $aggregationFactory,
        FilterableAttributes $filterableAttributes,
        AttributeEmptyBucketHandlerFactory $attributeEmptyBucketHandlerFactory
    ) {
        $this->searchResultFactory = $searchResultFactory;
        $this->aggregationFactory = $aggregationFactory;
        $this->filterableAttributes = $filterableAttributes;
        $this->attributeEmptyBucketHandlerFactory = $attributeEmptyBucketHandlerFactory;
    }

    /**
     * @return SearchResultInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function build(): SearchResultInterface
    {
        $searchResult = $this->searchResultFactory->create();
        $searchResult->setItems([]);
        $searchResult->setAggregations($this->getAggregations());
        $searchResult->setTotalCount(0);

        return $searchResult;
    }

    /**
     * @return AggregationInterface
     */
    private function getAggregations(): AggregationInterface
    {
        $emptyBuckets = [];
        $filterableAttributeCodes = $this->filterableAttributes->getMandatoryFilterableAttributesForLayeredNavigation();

        foreach ($filterableAttributeCodes as $attributeCode) {
            $attributeEmptyBucketHandler = $this->attributeEmptyBucketHandlerFactory->create(['attributeCode' => $attributeCode]);
            $emptyBuckets[$attributeEmptyBucketHandler->getBucketName()] = $attributeEmptyBucketHandler->getBucket();
        }

        return $this->aggregationFactory->create([
            'buckets' => $emptyBuckets
        ]);
    }
}
