<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor\FilterHandler;


class FilterHandlerFactory
{
    /**
     * @var RangeFilterHandlerFactory
     */
    private $rangeFilterHandlerFactory;

    /**
     * @var CategoryFilterHandlerFactory
     */
    private $categoryFilterHandlerFactory;

    /**
     * @var VisibilityFilterHandlerFactory
     */
    private $visibilityFilterHandlerFactory;

    /**
     * @var AttributeFilterHandlerFactory
     */
    private $attributeFilterHandlerFactory;

    /**
     * FilterHandlerFactory constructor.
     * @param RangeFilterHandlerFactory $rangeFilterHandlerFactory
     * @param CategoryFilterHandlerFactory $categoryFilterHandlerFactory
     * @param VisibilityFilterHandlerFactory $visibilityFilterHandlerFactory
     * @param AttributeFilterHandlerFactory $attributeFilterHandlerFactory
     */
    public function __construct(
        RangeFilterHandlerFactory $rangeFilterHandlerFactory,
        CategoryFilterHandlerFactory $categoryFilterHandlerFactory,
        VisibilityFilterHandlerFactory $visibilityFilterHandlerFactory,
        AttributeFilterHandlerFactory $attributeFilterHandlerFactory
    ) {
        $this->rangeFilterHandlerFactory = $rangeFilterHandlerFactory;
        $this->categoryFilterHandlerFactory = $categoryFilterHandlerFactory;
        $this->visibilityFilterHandlerFactory = $visibilityFilterHandlerFactory;
        $this->attributeFilterHandlerFactory = $attributeFilterHandlerFactory;
    }

    /**
     * @param string $filterKey
     * @param $filterValue
     * @return FilterHandlerInterface
     */
    public function resolve(string $filterKey, $filterValue): FilterHandlerInterface
    {
        if (is_array($filterValue) && !empty(array_intersect(array_keys($filterValue), ['from', 'to']))) {
            return $this->rangeFilterHandlerFactory->create(['filterKey' => $filterKey, 'filterValue' => $filterValue]);
        }

        switch ($filterKey) {
            case 'category_ids':
                $handler = $this->categoryFilterHandlerFactory->create(['filterKey' => $filterKey, 'filterValue' => $filterValue]);
                break;
            case 'visibility':
                $handler = $this->visibilityFilterHandlerFactory->create(['filterKey' => $filterKey, 'filterValue' => $filterValue]);
                break;
            default:
                $handler = $this->attributeFilterHandlerFactory->create(['filterKey' => $filterKey, 'filterValue' => $filterValue]);

        }

        return $handler;
    }
}
