<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\LiveSearchAdapter\Model\QueryArgumentProcessor\FilterHandler;

use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory;
use Magento\LiveSearchAdapter\Model\CategoryCache;

class CategoryFilterHandler implements FilterHandlerInterface
{
    /**
     * @var string
     */
    private $filterKey;

    /**
     * @var array
     */
    private $filterValue;

    /**
     * @var CollectionFactory
     */
    private $categoryCollectionFactory;

    /**
     * @var CategoryCache
     */
    private $categoryCache;

    /**
     * CategoryFilterHandler constructor.
     * @param string $filterKey
     * @param array $filterValue
     * @param CollectionFactory $categoryCollectionFactory
     * @param CategoryCache $categoryCache
     */
    public function __construct(
        string $filterKey,
        array $filterValue,
        CollectionFactory $categoryCollectionFactory,
        CategoryCache $categoryCache
    ) {
        $this->filterKey = $filterKey;
        $this->filterValue = $filterValue;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->categoryCache = $categoryCache;
    }
    /**
     * @return string
     */
    public function getFilterKey(): string
    {
        return 'categoryIds';
    }

    /**
     * @return array
     */
    public function getFilterValue(): array
    {
        $categoryIds = array_unique($this->filterValue);
        $categories = $this->getCategoryData($categoryIds);
        $parentCategoryIds = array_column($categories, 'parent_id');
        // filter out parent category ids, we only want to filter on child category
        $filteredCategoryIds = array_diff($categoryIds, $parentCategoryIds);
        return array_values($filteredCategoryIds);
    }

    /**
     * @param array $ids
     * @return array
     */
    private function getCategoryData(array $ids): array
    {
        $categoryIdsNotInCache = [];
        $categories = [];
        foreach ($ids as $id) {
            $categoryData = $this->categoryCache->load((int) $id);
            if ($categoryData) {
                $categories[] = $categoryData;
            } else {
                $categoryIdsNotInCache[] = $id;
            }
        }

        if (count($categoryIdsNotInCache) > 0) {
            $categoryCollection = $this->categoryCollectionFactory->create();
            $categoryCollection->getSelect()
                ->reset(\Zend_Db_Select::COLUMNS)
                ->columns(['entity_id', 'parent_id']);
            $categoriesNotInCache = $categoryCollection->addIdFilter($ids)->getData();
            foreach ($categoriesNotInCache as $categoryData) {
                $this->categoryCache->save((int)$categoryData['entity_id'], $categoryData);
                $categories[] = $categoryData;
            }
        }

        return $categories;
    }

    /**
     * @return string
     */
    public function getFilterType(): string
    {
        return 'in';
    }
}
