<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 */
declare(strict_types=1);

namespace Magento\InventoryCache\Plugin\InventoryIndexer\Indexer\Stock\Strategy\Sync;

use Magento\InventoryCache\Model\FlushCacheByProductIds;
use Magento\InventoryIndexer\Model\ResourceModel\GetProductIdsByStockIds;
use Magento\InventoryIndexer\Indexer\Stock\Strategy\Sync;

/**
 * Clean cache for specific products after non default stock reindex.
 */
class CacheFlush
{
    /**
     * @var FlushCacheByProductIds
     */
    private $flushCacheByProductIds;

    /**
     * @var GetProductIdsByStockIds
     */
    private $getProductIdsByStockIds;

    /**
     * @param FlushCacheByProductIds $flushCacheByProductIds
     * @param GetProductIdsByStockIds $getProductIdsForCacheFlush
     */
    public function __construct(
        FlushCacheByProductIds $flushCacheByProductIds,
        GetProductIdsByStockIds $getProductIdsForCacheFlush
    ) {
        $this->flushCacheByProductIds = $flushCacheByProductIds;
        $this->getProductIdsByStockIds = $getProductIdsForCacheFlush;
    }

    /**
     * Clean cache after non default stock reindex.
     *
     * @param Sync $subject
     * @param callable $proceed
     * @param array $stockIds
     * @return void
     * @throws \Exception in case product entity type hasn't been initialize.
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function aroundExecuteList(Sync $subject, callable $proceed, array $stockIds)
    {
        $beforeReindexProductIds = $this->getProductIdsByStockIds->execute($stockIds);
        $proceed($stockIds);
        $afterReindexProductIds = $this->getProductIdsByStockIds->execute($stockIds);
        $productIdsForCacheClean = array_diff($beforeReindexProductIds, $afterReindexProductIds);
        if ($productIdsForCacheClean) {
            $this->flushCacheByProductIds->execute($productIdsForCacheClean);
        }
    }
}
