<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\Elasticsearch\Model\ResourceModel;

use Magento\Eav\Api\Data\AttributeInterface;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Eav\Model\Config;
use Magento\Catalog\Api\Data\ProductAttributeInterface;
use Magento\Framework\EntityManager\MetadataPool;

/**
 * Elasticsearch index resource model.
 */
class Index extends \Magento\AdvancedSearch\Model\ResourceModel\Index
{
    /**
     * @var ProductRepositoryInterface
     */
    protected $productRepository;

    /**
     * @var CategoryRepositoryInterface
     */
    protected $categoryRepository;

    /**
     * @var Config
     */
    protected $eavConfig;

    /**
     * @var array
     */
    private $attributeData = [];

    /**
     * Index constructor.
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param MetadataPool $metadataPool
     * @param ProductRepositoryInterface $productRepository
     * @param CategoryRepositoryInterface $categoryRepository
     * @param Config $eavConfig
     * @param null $connectionName
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        MetadataPool $metadataPool,
        ProductRepositoryInterface $productRepository,
        CategoryRepositoryInterface $categoryRepository,
        Config $eavConfig,
        $connectionName = null
    ) {
        $this->productRepository = $productRepository;
        $this->categoryRepository = $categoryRepository;
        $this->eavConfig = $eavConfig;
        parent::__construct($context, $storeManager, $metadataPool, $connectionName);
    }

    /**
     * Retrieve all attributes for given product ids.
     *
     * @param int $productId
     * @param array $indexData
     * @return array
     */
    public function getFullProductIndexData($productId, $indexData)
    {
        $productAttributes = [];
        $attributeCodes = $this->eavConfig->getEntityAttributeCodes(ProductAttributeInterface::ENTITY_TYPE_CODE);
        $product = $this->productRepository->getById($productId);
        foreach ($attributeCodes as $attributeCode) {
            $value = $product->getData($attributeCode);
            $attributeData = $this->getAttributeData($attributeCode);
            $attributeId = $attributeData[AttributeInterface::ATTRIBUTE_ID];
            if (array_key_exists($attributeId, $indexData)) {
                if (is_array($indexData[$attributeId])) {
                    if (isset($indexData[$attributeId][$productId])) {
                        $value = $indexData[$attributeId][$productId];
                    } else {
                        $value = array_values($indexData[$attributeId]);
                    }
                } else {
                    $value = $indexData[$attributeId];
                }
            }

            if ($value) {
                $productAttributes[$attributeCode] = $value;

                if (in_array($attributeData[AttributeInterface::FRONTEND_INPUT], ['select', 'multiselect'])) {
                    $valueLabel = $this->getOptionsLabel($attributeData, $value);
                    $productAttributes[$attributeCode . '_value'] = $valueLabel;
                }
            }
        }

        return $productAttributes;
    }

    /**
     * Prepare full category index data for products.
     *
     * @param int $storeId
     * @param null|array $productIds
     * @return array
     */
    public function getFullCategoryProductIndexData($storeId = null, $productIds = null)
    {
        $categoryPositions = $this->getCategoryProductIndexData($storeId, $productIds);
        $categoryData = [];

        foreach ($categoryPositions as $productId => $positions) {
            foreach ($positions as $categoryId => $position) {
                $category = $this->categoryRepository->get($categoryId, $storeId);
                $categoryName = $category->getName();
                $categoryData[$productId][] = [
                    'id' => $categoryId,
                    'name' => $categoryName,
                    'position' => $position
                ];
            }
        }

        return $categoryData;
    }

    /**
     * Returns attribute options label as one string.
     *
     * Retrieve attribute selected options label as string concatenated by space.
     * Used to return value for dropdown attribute '_value' index field.
     *
     * @param array $attributeData
     * @param mixed $value
     * @return string
     */
    private function getOptionsLabel(
        array $attributeData,
        $value
    ) {
        $attributeValues = is_array($value) ? $value : explode(',', $value);
        $valueLabel = '';

        $options = $attributeData[AttributeInterface::OPTIONS];
        foreach ($attributeValues as $attributeValue) {
            if (isset($options[$attributeValue])) {
                $valueLabel .= $options[$attributeValue] . ' ';
            }
        }

        return rtrim($valueLabel);
    }

    /**
     * Get product attribute data by attribute id
     *
     * @param string $attributeCode
     * @return array
     */
    private function getAttributeData($attributeCode)
    {
        if (!array_key_exists($attributeCode, $this->attributeData)) {
            $attribute = $this->eavConfig->getAttribute(
                ProductAttributeInterface::ENTITY_TYPE_CODE,
                $attributeCode
            );

            $options = [];
            if ($attribute->getFrontendInput() === 'select' || $attribute->getFrontendInput() === 'multiselect') {
                foreach ($attribute->getOptions() as $option) {
                    $options[$option->getValue()] = $option->getLabel();
                }
            }
            $this->attributeData[$attributeCode] = [
                AttributeInterface::ATTRIBUTE_ID => $attribute->getAttributeId(),
                AttributeInterface::ATTRIBUTE_CODE => $attribute->getAttributeCode(),
                AttributeInterface::FRONTEND_INPUT => $attribute->getFrontendInput(),
                AttributeInterface::BACKEND_TYPE => $attribute->getBackendType(),
                AttributeInterface::OPTIONS => $options,
            ];
        }

        return $this->attributeData[$attributeCode];
    }
}
