<?php
/**
 * Copyright © 2013-2017 Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\Elasticsearch\Plugin\Model\ResourceModel\Attribute;

use Magento\Catalog\Api\Data\EavAttributeInterface;
use Magento\Elasticsearch\Model\Adapter\FieldType;
use Magento\Elasticsearch\Model\Adapter\Index\IndexNameResolver;
use Magento\Elasticsearch\Model\Config;
use Magento\Elasticsearch\SearchAdapter\ConnectionManager;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Refresh attribute mapping in search indexer.
 */
class UpdateMapping
{
    /**
     * @var IndexNameResolver
     */
    private $indexNameResolver;

    /**
     * @var FieldType
     */
    private $fieldType;

    /**
     * @var ConnectionManager
     */
    private $connectionManager;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Config
     */
    private $clientConfig;

    /**
     * @param IndexNameResolver $indexNameResolver
     * @param ConnectionManager $connectionManager
     * @param StoreManagerInterface $storeManager
     * @param FieldType $fieldType
     * @param Config $clientConfig
     */
    public function __construct(
        IndexNameResolver $indexNameResolver,
        ConnectionManager $connectionManager,
        StoreManagerInterface $storeManager,
        FieldType $fieldType,
        Config $clientConfig
    ) {
        $this->indexNameResolver = $indexNameResolver;
        $this->connectionManager = $connectionManager;
        $this->storeManager = $storeManager;
        $this->fieldType = $fieldType;
        $this->clientConfig = $clientConfig;
    }

    /**
     * Refresh indexer mapping on attribute save (searchable flag change).
     *
     * @param \Magento\Catalog\Model\ResourceModel\Attribute $subject
     * @param \Closure $proceed
     * @param \Magento\Framework\Model\AbstractModel $attribute
     *
     * @return \Magento\Catalog\Model\ResourceModel\Attribute
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function aroundSave(
        \Magento\Catalog\Model\ResourceModel\Attribute $subject,
        \Closure $proceed,
        \Magento\Framework\Model\AbstractModel $attribute
    ) {
        $updateRequired = $this->isUpdateRequired($attribute);

        $result = $proceed($attribute);

        if ($updateRequired && $this->clientConfig->isElasticsearchEnabled()) {
            $mappedIndexerId = $this->indexNameResolver->getIndexMapping(
                \Magento\CatalogSearch\Model\Indexer\Fulltext::INDEXER_ID
            );

            $attribute = $this->getAttributeMapping($attribute);
            $client = $this->connectionManager->getConnection();
            $storeIds = array_keys($this->storeManager->getStores());

            /** @var int $storeId */
            foreach ($storeIds as $storeId) {
                $indexName = $this->indexNameResolver->getIndexName($storeId, $mappedIndexerId, []);

                $client->addFieldsMapping(
                    $attribute,
                    $indexName,
                    $this->clientConfig->getEntityType()
                );
            }
        }

        return $result;
    }

    /**
     * Search indexer attribute mapping should be updated.
     *
     * Checks if mapping should be updated for attribute according to fields,
     * that are used by search indexer to mark field as "indexable".
     *
     * @param \Magento\Framework\Model\AbstractModel $model
     * @return bool
     */
    private function isUpdateRequired(\Magento\Framework\Model\AbstractModel $model)
    {
        $result = false;
        if ($model->isObjectNew()) {
            $fields = [
                EavAttributeInterface::IS_SEARCHABLE,
                EavAttributeInterface::IS_FILTERABLE,
                EavAttributeInterface::IS_VISIBLE_IN_ADVANCED_SEARCH,
            ];

            foreach ($fields as $fieldName) {
                if ($model->dataHasChangedFor($fieldName) && (bool)$model->getData($fieldName)) {
                    $result = true;
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Returns attribute mapping.
     *
     * Returns attribute mapping by backend type. In case of select/multiselect
     * attribute also returns additional "label" field mapping.
     *
     * @param \Magento\Framework\Model\AbstractModel $attribute
     * @return array
     */
    private function getAttributeMapping(\Magento\Framework\Model\AbstractModel $attribute)
    {
        $attributeCode = $attribute->getAttributeCode();
        $mapping[$attributeCode] = [
            'type' => $this->fieldType->getFieldType($attribute),
        ];

        if (in_array($attribute->getFrontendInput(), ['select', 'multiselect'], true)) {
            $mapping[$attributeCode . '_value'] = [
                'type' => FieldType::ES_DATA_TYPE_STRING,
            ];
        }

        return $mapping;
    }
}
