<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\DataServices\ViewModel\Checkout;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\View\Element\Block\ArgumentInterface;

/**
 * ViewModel for Checkout Success Context
 */
class SuccessContextProvider implements ArgumentInterface
{
    /**
     * @var CheckoutSession
     */
    private CheckoutSession $checkoutSession;

    /**
     * @var Json
     */
    private Json $jsonSerializer;

    /**
     * @var PriceCurrencyInterface
     */
    private PriceCurrencyInterface $priceCurrency;

    /**
     * @param CheckoutSession $checkoutSession
     * @param Json $jsonSerializer
     * @param PriceCurrencyInterface $priceCurrency
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        Json $jsonSerializer,
        PriceCurrencyInterface $priceCurrency
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->jsonSerializer = $jsonSerializer;
        $this->priceCurrency = $priceCurrency;
    }

    /**
     * Return cart id for event tracking
     *
     * @return int
     */
    public function getCartId(): int
    {
        return (int)$this->checkoutSession->getLastRealOrder()->getQuoteId();
    }

    /**
     * Return customer email for event tracking
     *
     * @return string
     */
    public function getCustomerEmail(): string
    {
        return (string)$this->checkoutSession->getLastRealOrder()->getCustomerEmail();
    }

    /**
     * Return order id for event tracking
     *
     * @return string
     */
    public function getOrderId(): string
    {
        return (string)$this->checkoutSession->getLastRealOrder()->getIncrementId();
    }

    /**
     * Return order subtotal rounded to 2 decimals
     * 
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return float
     */
    private function getOrderSubTotal($order): float
    {
        return round((float)$order->getSubTotal(), 2);
    }

    /**
     * Return payment method data
     *
     * @return string
     */
    public function getPayment(): string
    {
        $order = $this->checkoutSession->getLastRealOrder();
        $paymentData = [
            'total' => $this->getOrderSubTotal($order),
        ];

        $payment = $order->getPayment();
        // Payment might not exist for async placed orders.
        if ($payment) {
            $paymentData['paymentMethodCode'] = $payment->getMethod();
            $paymentData['paymentMethodName'] = $payment->getMethodInstance()->getTitle();
        }

        return $this->jsonSerializer->serialize([$paymentData]);
    }

    /**
     * Return shipping data
     *
     * @return string
     */
    public function getShipping(): string
    {
        return $this->jsonSerializer->serialize(
            [
                'shippingMethod' => $this->checkoutSession->getLastRealOrder()->getShippingMethod(),
                'shippingAmount' => $this->checkoutSession->getLastRealOrder()->getShippingAmount(),
            ]
        );
    }
}
