<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\DataServicesGraphQl\Model\Resolver;

use Magento\Framework\App\CacheInterface;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\ServicesId\Model\ServicesConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use Magento\Framework\Locale\CurrencyInterface;
use Magento\CatalogInventory\Model\Configuration as InventoryConfiguration;
use Magento\Search\Model\Query as SearchQuery;

/**
 * Class that returns store configuration context for dataServicesStoreConfigurationContext
 */
class StoreConfigurationContext extends Template implements ResolverInterface
{

    /**
     * Extension constants
     */
    const CATALOG_EXPORTER_MODULE_NAME = 'Magento/CatalogDataExporter';
    const CATALOG_EXPORTER_PACKAGE_NAME = 'magento/saas-export';
    const CATALOG_EXPORTER_VERSION_CACHE_TAG = 'catalog_exporter_extension_version';

    /**
     * Autocomplete limit
     *
     * @var string
     */
    private const AUTOCOMPLETE_LIMIT = 'catalog/search/autocomplete_limit';

    /**
     * Results per page options
     *
     * @var string
     */
    private const RESULTS_PER_PAGE_OPTIONS = 'catalog/frontend/grid_per_page_values';

    /**
     * Results per page default
     *
     * @var string
     */
    private const RESULTS_PER_PAGE_DEFAULT = 'catalog/frontend/grid_per_page';

    /**
     * locale language
     *
     * @var string
     */
    private const LOCALE_LANGUAGE = 'general/locale/code';

    /**
     * Whether to show "All" option in the "Show X Per Page" dropdown
     *
     * @var string
     */
    private const ALLOW_ALL_PRODUCTS = 'catalog/frontend/list_allow_all';

    /**
     * @var CurrencyInterface
     */
    private CurrencyInterface $localeCurrency;

    /**
     * @var ScopeConfigInterface
     */
    private ScopeConfigInterface $scopeConfig;

    /**
     * StoreConfigurationContext constructor.
     * @param Context $context
     * @param ServicesConfigInterface $servicesConfig
     * @param CacheInterface $cache
     * @param ScopeConfigInterface $scopeConfig
     * @param CurrencyInterface $localeCurrency,
     */
    public function __construct(
        Context $context,
        ServicesConfigInterface $servicesConfig,
        CacheInterface $cache,
        ScopeConfigInterface $scopeConfig,
        CurrencyInterface $localeCurrency,
    ) {
        $this->servicesConfig = $servicesConfig;
        $this->cache = $cache;
        $this->scopeConfig = $scopeConfig;
        $this->localeCurrency = $localeCurrency;
        parent::__construct($context);
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        ?array $value = null,
        ?array $args = null
    ) {
        return [
          'currency_symbol' => $this->getCurrencySymbol(),
          'currency_code' => $this->getCurrencyCode(),
          'currency_rate' => $this->getCurrencyRate(),
          'page_size' => $this->getAutocompleteLimit(),
          'page_size_options' => $this->getResultsPerPageOptions(),
          'default_page_size_option' => $this->getResultsPerPageDefault(),
          'display_out_of_stock' => $this->isDisplayOutOfStock(),
          'allow_all_products' => $this->isAllowAllProducts(),
          'locale' => $this->getLanguageCode(),
          'min_query_length' => $this->getMinQueryLength(),
      ];
    }

    /**
     * Returns autocomplete limit from config
     *
     * @return int
     */
    public function getAutocompleteLimit(): int
    {
        return (int)$this->scopeConfig->getValue(
            self::AUTOCOMPLETE_LIMIT,
            ScopeInterface::SCOPE_STORES
        );
    }

    /**
     * Returns results per page options from config
     *
     * @return string
     */
    public function getResultsPerPageOptions(): string
    {
        return (string)$this->scopeConfig->getValue(
            self::RESULTS_PER_PAGE_OPTIONS,
            ScopeInterface::SCOPE_STORES
        );
    }

    /**
     * Returns results per page default from config
     *
     * @return string
     */
    public function getResultsPerPageDefault(): string
    {
        return (string)$this->scopeConfig->getValue(
            self::RESULTS_PER_PAGE_DEFAULT,
            ScopeInterface::SCOPE_STORES
        );
    }

    /**
     * Returns whether to allow all products in results per page options
     *
     * @return bool
     */
    public function isAllowAllProducts(): bool
    {
        return (bool)$this->scopeConfig->getValue(
            self::ALLOW_ALL_PRODUCTS,
            ScopeInterface::SCOPE_STORES
        );
    }

    /**
     * Returns locale language from config
     *
     * @return string
     */
    public function getLanguageCode(): string
    {
        return (string)$this->scopeConfig->getValue(
            self::LOCALE_LANGUAGE,
            ScopeInterface::SCOPE_STORES
        );
    }

    /**
     * Returns display out of stock from config
     *
     * @return bool
     */
    public function isDisplayOutOfStock(): bool
    {
        return (bool) $this->scopeConfig->getValue(InventoryConfiguration::XML_PATH_SHOW_OUT_OF_STOCK);
    }

    /**
     * Get store currency symbol
     *
     * @return string
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @throws CurrencyException
     */
    public function getCurrencySymbol(): string
    {
        $currencyCode = $this->getCurrencyCode();
        $currency = $this->localeCurrency->getCurrency($currencyCode);
        $currencySymbol = $currency->getSymbol() ?: $currency->getShortName();

        if (empty($currencySymbol)) {
            return $currencyCode;
        }

        return $currencySymbol;
    }

    /**
     * Get store currency code
     *
     * @return string
     */
    public function getCurrencyCode(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $currencyCode = $this->_storeManager->getStore($storeId)->getCurrentCurrency()->getCurrencyCode();

        return $currencyCode;
    }

    /**
     * Get store currency conversion rate
     *
     * @return string
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @throws CurrencyException
     */
    public function getCurrencyRate(): string
    {
        $storeId = $this->getRequest()->getParam('store');
        $store = $this->_storeManager->getStore($storeId);
        $currencyCode = $store->getCurrentCurrency()->getCurrencyCode();
        $currencyRate = $store->getBaseCurrency()->getRate($currencyCode);

        return (string) $currencyRate;
    }

    /**
     * Retrieve minimum query length
     *
     * @return string
     */
    public function getMinQueryLength(): string
    {
        return $this->scopeConfig->getValue(
            SearchQuery::XML_PATH_MIN_QUERY_LENGTH,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }
}
