<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\DataServicesGraphQl\Model\Resolver;

use Magento\DataServices\Model\VersionFinderInterface;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\UrlInterface;
use Magento\ServicesId\Model\ServicesConfigInterface;

/**
 * Class that implements the dataServicesStorefrontInstanceContext resolver
 */
class StorefrontInstanceContext implements ResolverInterface
{

    /**
     * Extension constants
     */
    const CATALOG_EXPORTER_MODULE_NAME = 'Magento/CatalogDataExporter';
    const CATALOG_EXPORTER_PACKAGE_NAME = 'magento/saas-export';
    const CATALOG_EXPORTER_VERSION_CACHE_TAG = 'catalog_exporter_extension_version';

    /**
     * @var VersionFinderInterface
     */
    private $versionFinder;

    /**
     * @var ServicesConfigInterface
     */
    private $servicesConfig;

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * StorefrontInstanceContext constructor.
     * @param VersionFinderInterface $versionFinder
     * @param ServicesConfigInterface $servicesConfig
     * @param CacheInterface $cache
     */
    public function __construct(
        VersionFinderInterface $versionFinder,
        ServicesConfigInterface $servicesConfig,
        CacheInterface $cache
    ) {
        $this->versionFinder = $versionFinder;
        $this->servicesConfig = $servicesConfig;
        $this->cache = $cache;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        $store = $context->getExtensionAttributes()->getStore();
        return [
            'environment_id' => $this->servicesConfig->getEnvironmentId(),
            'environment' => $this->servicesConfig->getEnvironmentType(),
            'ims_org_id' => $this->servicesConfig->getImsOrganizationId(),
            'store_url' => $store->getBaseUrl(UrlInterface::URL_TYPE_WEB),
            'website_id' => $store->getWebsiteId(),
            'website_code' => $store->getWebsite()->getCode(),
            'store_id' => $store->getGroup()->getId(),
            'store_code' => $store->getGroup()->getCode(),
            'store_view_id' => $store->getId(),
            'store_view_code' => $store->getCode(),
            'website_name' => $store->getWebsite()->getName(),
            'store_name' => $store->getGroup()->getName(),
            'store_view_name' => $store->getName(),
            'catalog_extension_version' => $this->getCatalogExtensionVersion(),
            'base_currency_code' => $store->getBaseCurrencyCode(),
            'store_view_currency_code' => $store->getCurrentCurrency()->getCode()
        ];
    }

    /**
     * Return catalog extension version
     *
     * @return string|null
     */
    private function getCatalogExtensionVersion(): ?string
    {
        $catalogVersion = $this->cache->load(self::CATALOG_EXPORTER_VERSION_CACHE_TAG);
        if (!$catalogVersion) {
            $catalogVersion = $this->versionFinder->getVersionFromComposer(self::CATALOG_EXPORTER_PACKAGE_NAME)
                ?: $this->versionFinder->getVersionFromFiles(
                    self::CATALOG_EXPORTER_MODULE_NAME,
                    self::CATALOG_EXPORTER_PACKAGE_NAME
                );
            $this->cache->save($catalogVersion, self::CATALOG_EXPORTER_VERSION_CACHE_TAG);
        }

        return $catalogVersion ?? null;
    }
}
