<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Model;

use Magento\Framework\Api\CustomAttributesDataInterface;
use Magento\Framework\Api\AttributeInterface;
use Magento\Framework\Api\AttributeTypeResolverInterface;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Processes custom attributes and produces an array for the data.
 */
class ExtendedCustomAttributesProcessor
{
    public function __construct(
        private DataObjectProcessor $dataObjectProcessor,
        private AttributeTypeResolverInterface $attributeTypeResolver
    ) {
    }

    /**
     * Writes out the custom attributes for a given object into an array.
     *
     * @param CustomAttributesDataInterface $objectWithCustomAttributes
     * @param string $dataObjectType
     * @return array
     */
    public function buildOutputDataArray(CustomAttributesDataInterface $objectWithCustomAttributes, $dataObjectType): array
    {
        $customAttributes = $objectWithCustomAttributes->getCustomAttributes();
        $result = [];
        foreach ($customAttributes as $customAttribute) {
            $result[] = $this->convertCustomAttribute($customAttribute, $dataObjectType);
        }
        return $result;
    }

    /**
     * Convert custom_attribute object to use flat array structure
     *
     * @param AttributeInterface $customAttribute
     * @param string $dataObjectType
     * @return array
     */
    private function convertCustomAttribute(AttributeInterface $customAttribute, $dataObjectType): array
    {
        $data = [
            AttributeInterface::ATTRIBUTE_CODE => $customAttribute->getAttributeCode()
        ];
        $value = $customAttribute->getValue();
        if (is_object($value)) {
            $value = $this->processObjectValue($value, $customAttribute->getAttributeCode(), $dataObjectType);
        } elseif (is_array($value)) {
            $valueResult = [];
            foreach ($value as $objectKey => $singleValue) {
                if (is_object($singleValue)) {
                    $singleValue = $this->processObjectValue(
                        $singleValue,
                        $customAttribute->getAttributeCode(),
                        $dataObjectType
                    );
                }
                $valueResult[$objectKey] = $singleValue;
            }
            $value = $valueResult;
        }
        $data[AttributeInterface::VALUE] = $value;
        return $data;
    }

    /**
     * Process object values in custom attributes
     *
     * @param object $objectValue
     * @param string $attributeCode
     * @param string $dataObjectType
     * @return array
     */
    private function processObjectValue(object $objectValue, string $attributeCode, $dataObjectType): array
    {
        $type = $this->attributeTypeResolver->resolveObjectType(
            $attributeCode,
            $objectValue,
            $dataObjectType
        );
        return $this->dataObjectProcessor->buildOutputDataArray($objectValue, $type);
    }
}
